<?php
/*
Plugin Name: Agrégateur (ACF) - XML Corrigé (v1.6 + Cotations) - filtrage conseillers
Description: Import XML complet - Clients + Contrats + Cotations (ajout code ISIN + plus-values totales + fix catégories & update contrats + ajout nom/prénom contrat). Filtre par codes conseillers via répéteur option ACF 'codes_conseillers' -> champ 'code_conseiller'.
Version: 1.6.1
Author: Kyllian GALLAY (modifié)
*/

if (!defined('ABSPATH')) exit;

/* ========= Helpers globaux (déclarés si non existants) ========= */
if (!function_exists('_agr_ensure_category')) {
    function _agr_ensure_category($name) {
        if (!$name) return 0;
        $term = term_exists($name, 'category');
        if ($term === 0 || $term === null) {
            $t = wp_insert_term($name, 'category');
            if (!is_wp_error($t) && isset($t['term_id'])) return (int)$t['term_id'];
            return 0;
        }
        return is_array($term) ? (int)$term['term_id'] : (int)$term;
    }
}

if (!function_exists('_agr_remove_new_labels')) {
    function _agr_remove_new_labels() {
        $newClientId = term_exists('Nouveau client', 'category');
        $newContractId = term_exists('Nouveau contrat', 'category');

        if ($newClientId) {
            $client_ids = get_posts(['post_type'=>'clients','posts_per_page'=>-1,'fields'=>'ids']);
            foreach ($client_ids as $cid) {
                @wp_remove_object_terms($cid, 'Nouveau client', 'category');
            }
        }
        if ($newContractId) {
            $contract_ids = get_posts(['post_type'=>'contrats','posts_per_page'=>-1,'fields'=>'ids']);
            foreach ($contract_ids as $cid) {
                @wp_remove_object_terms($cid, 'Nouveau contrat', 'category');
            }
        }
    }
}

if (!function_exists('format_euro')) {
    function format_euro($v){return number_format((float)$v, 2, ',', ' ') . ' €';}
}
if (!function_exists('format_pct')) {
    function format_pct($v){return number_format((float)$v, 2, ',', ' ') . ' %';}
}
if (!function_exists('format_date_fr')) {
    function format_date_fr($d){if(!$d)return ''; $ts=strtotime($d); return $ts?date('d/m/Y',$ts):$d;}
}
if (!function_exists('parse_number')) {
    function parse_number($s){
        if ($s === null) return 0.0;
        $s = (string)$s;
        $s = str_replace(["\xc2\xa0"," "], '', $s);
        $s = str_replace(['€','€'], '', $s);
        $s = str_replace(',', '.', $s);
        $s = preg_replace('/[^\d\.\-]/', '', $s);
        return $s === '' ? 0.0 : floatval($s);
    }
}

add_action('init', function () {

    if (!isset($_GET['agr'])) return;

    if (!is_user_logged_in() || !current_user_can('administrator')) {
        status_header(403);
        echo "Forbidden";
        exit;
    }

    $cmd = sanitize_text_field($_GET['agr']);

    if ($cmd === 'start') {
        do_action('agr_queue_start');
        echo "OK start";
        exit;
    }

    if ($cmd === 'next') {
        do_action('agr_queue_process_next');
        echo "OK next";
        exit;
    }

    if ($cmd === 'run') {
        do_action('agr_queue_start');
        do_action('agr_queue_process_next');
        echo "OK run";
        exit;
    }

    echo "Unknown command. Use ?agr=start | ?agr=next | ?agr=run";
    exit;
});
/* ========= CRON / QUEUE IMPORT ========= */

// 1) Dossier inbox (uploads/agregateur-inbox)
if (!defined('AGR_INBOX_DIR')) {
    $u = wp_upload_dir();
    define('AGR_INBOX_DIR', trailingslashit($u['basedir']) . 'agregateur-inbox/');
}

// 2) Réglages (tu peux changer ici)
if (!defined('AGR_PROCESS_EVERY_SECONDS')) define('AGR_PROCESS_EVERY_SECONDS', 300); // 5 minutes
if (!defined('AGR_TEST_DELAY_SECONDS'))    define('AGR_TEST_DELAY_SECONDS', 300);   // test: dans 5 minutes

// 3) Intervalle custom "every_5_minutes"
add_filter('cron_schedules', function($schedules){
    if (!isset($schedules['every_5_minutes'])) {
        $schedules['every_5_minutes'] = [
            'interval' => AGR_PROCESS_EVERY_SECONDS,
            'display'  => 'Every 5 minutes'
        ];
    }
    if (!isset($schedules['weekly'])) {
        $schedules['weekly'] = [
            'interval' => 7 * DAY_IN_SECONDS,
            'display'  => 'Weekly'
        ];
    }
    return $schedules;
});

// 4) Installer le cron (weekly mer 09:00) + test dans 5 min
register_activation_hook(__FILE__, function(){
    // Crée le dossier inbox si absent
    if (!file_exists(AGR_INBOX_DIR)) {
        wp_mkdir_p(AGR_INBOX_DIR);
        // protection basique
        @file_put_contents(AGR_INBOX_DIR . 'index.html', '');
        @file_put_contents(AGR_INBOX_DIR . '.htaccess', "Order allow,deny\nDeny from all\n");
    }

    // --- TEST : déclenche une campagne dans 5 minutes ---
    if (!wp_next_scheduled('agr_queue_start')) {
        wp_schedule_single_event(time() + AGR_TEST_DELAY_SECONDS, 'agr_queue_start');
    }

    // --- PROD : tous les mercredis à 09:00 (heure WP) ---
    // Calcule le prochain mercredi 09:00
    $tz = wp_timezone();
    $now = new DateTime('now', $tz);
    $next = clone $now;
    $next->setTime(9, 0, 0);

    // Mercredi = 3 (ISO-8601: 1=lundi ... 3=mercredi)
    $day = (int)$next->format('N');
    $delta = (3 - $day + 7) % 7;
    if ($delta === 0 && $now > $next) $delta = 7;
    $next->modify("+{$delta} days");

    $ts = $next->getTimestamp();

    if (!wp_next_scheduled('agr_queue_start_weekly')) {
        wp_schedule_event($ts, 'weekly', 'agr_queue_start_weekly');
    }
});

register_deactivation_hook(__FILE__, function(){
    // Nettoyage événements
    $t = wp_next_scheduled('agr_queue_start');
    if ($t) wp_unschedule_event($t, 'agr_queue_start');

    $t = wp_next_scheduled('agr_queue_start_weekly');
    if ($t) wp_unschedule_event($t, 'agr_queue_start_weekly');

    $t = wp_next_scheduled('agr_queue_process_next');
    if ($t) wp_unschedule_event($t, 'agr_queue_process_next');
});

// 5) Démarre la campagne (test)
add_action('agr_queue_start', function(){
    error_log("[AGRÉGATEUR][CRON] 🔔 Démarrage campagne (TEST) - scan inbox");
    agr_start_processing_loop();
});

// 6) Démarre la campagne (weekly)
add_action('agr_queue_start_weekly', function(){
    error_log("[AGRÉGATEUR][CRON] 🔔 Démarrage campagne (WEEKLY) - scan inbox");
    agr_start_processing_loop();
});

// 7) Lance le loop 1 fichier / 5 min
function agr_start_processing_loop(){
    if (!file_exists(AGR_INBOX_DIR)) return;

    // Si déjà planifié, ne rien faire
    if (!wp_next_scheduled('agr_queue_process_next')) {
        wp_schedule_event(time() + 10, 'every_5_minutes', 'agr_queue_process_next');
    }
}

// 8) Traitement : 1 fichier à chaque tick
add_action('agr_queue_process_next', function(){
$files = glob(AGR_INBOX_DIR . '*.xml') ?: [];
$files = array_values(array_filter($files, function($p){
    $b = basename($p);
    return stripos($b, 'DONE_') !== 0 && stripos($b, 'FAILED_') !== 0;
}));

if (!$files) {
        // plus rien => stop le cron
        $t = wp_next_scheduled('agr_queue_process_next');
        if ($t) wp_unschedule_event($t, 'agr_queue_process_next');
        error_log("[AGRÉGATEUR][CRON] ✅ Inbox vide, arrêt du loop");
        return;
    }

    // prend le plus ancien
    usort($files, function($a,$b){
        return filemtime($a) <=> filemtime($b);
    });

    $file_path = $files[0];
    $base = basename($file_path);

    // Compagnie = préfixe avant "_"
    $compagnie = '';
    if (strpos($base, '_') !== false) {
        $compagnie = sanitize_text_field(substr($base, 0, strpos($base, '_')));
    } else {
        // fallback si pas de "_"
        $compagnie = sanitize_text_field(pathinfo($base, PATHINFO_FILENAME));
    }

    error_log("[AGRÉGATEUR][CRON] ▶️ Import fichier: {$base} | compagnie={$compagnie}");

    // ✅ Import : agr_import_xml_file() retourne un tableau => on teste success
 $res = agr_import_xml_file($file_path, $compagnie, false);

    $ok  = is_array($res) && !empty($res['success']);

    if ($ok) {
        // (optionnel) log des compteurs
        if (!empty($res['counters'])) {
            error_log("[AGRÉGATEUR][CRON] Counters: " . json_encode($res['counters']));
        }

        // archive ou suppression
        @rename($file_path, AGR_INBOX_DIR . 'DONE_' . $base);

        error_log("[AGRÉGATEUR][CRON] ✅ Import OK, fichier archivé: DONE_{$base}");

    } else {
        // pour éviter boucle infinie sur un fichier cassé : on le renomme
        @rename($file_path, AGR_INBOX_DIR . 'FAILED_' . $base);

        $msg = (is_array($res) && !empty($res['message'])) ? (string)$res['message'] : 'unknown error';
        error_log("[AGRÉGATEUR][CRON] ❌ Import FAIL ({$msg}), renommé en FAILED_: {$base}");
    }
});





/* ========= Shortcode d'import (plugin) ========= */
add_shortcode('import_xml_agregateur', 'import_xml_agregateur_shortcode');

function agr_should_log_history($compagnie) {
    $s = agr_get_settings();
    if (empty($s['enable_history'])) return false;
    return in_array((string)$compagnie, ['SwissLife', 'Abeille'], true);
}

function agr_push_history($message_html) {
    if (!function_exists('get_field')) return;
    $historique = get_field('historique_rep', 'option') ?: [];
    $historique[] = [
        'historique_sous' => $message_html,
        'timestamp'       => current_time('Y-m-d H:i:s'),
    ];
    update_field('historique_rep', $historique, 'option');
}

function agr_history_msg($text) {
    return '<span style="margin-right:12px;">💻</span>'
         . '<span style="margin-right:8px;">' . $text . '</span>';
}

function agr_get_settings() {
    $defaults = [
        'import_clients'   => 1,
        'import_contrats'  => 1,
        'import_cotations' => 1,
        'enable_history'   => 1,
        'disable_heavy_hooks' => 1,
    ];
    $opts = get_option('agr_settings', []);
    if (!is_array($opts)) $opts = [];
    return array_merge($defaults, $opts);
}

function agr_update_settings($new) {
    $current = agr_get_settings();
    $merged = array_merge($current, $new);
    update_option('agr_settings', $merged, false);
}


function agr_company_profile($compagnie) {
    $compagnie = (string)$compagnie;

    // Profil par défaut = SwissLife (comportement actuel)
    $profiles = [
        'SwissLife' => [
            // personne
            'person_addr_attr' => 'adr_ligne1',

            // supports
            // false = ref_ID_support_info_generale = un ID interne, et l’ISIN vient de support_info_gen_consomme@code_courant
            'support_id_is_isin' => false,

            // lookup support_info_gen_consomme
            'support_lookup_attr' => 'ID_support_info_generale', // attribut utilisé dans le xpath pour matcher
            'support_isin_attr'   => 'code_courant',             // attribut qui contient l’ISIN
            'support_label_attr'  => 'designation',              // attribut libellé
        ],

        'Abeille' => [
            // DIFF 1: adr_ligne1 -> adr_ligne3
            'person_addr_attr' => 'adr_ligne3',

            // DIFF 2: ref_ID_support_info_generale est déjà l’ISIN
            // DIFF 3: ID_support_info_generale idem (si on fait quand même une lookup, l’ID = ISIN)
            'support_id_is_isin' => true,

            // Dans ton énoncé tu dis “ID_support_info_generale -> même chose”
            // Donc le lookup par ID devrait aussi marcher si le node existe.
            'support_lookup_attr' => 'ID_support_info_generale',
            'support_isin_attr'   => 'code_courant',   // pas indispensable si support_id_is_isin = true
            'support_label_attr'  => 'designation',
        ],
    ];

    return $profiles[$compagnie] ?? $profiles['SwissLife'];
}


add_action('admin_menu', function () {
    add_menu_page(
        'Agrégateur',
        'Agrégateur',
        'manage_options',
        'agregateur-dashboard',
        'agr_admin_dashboard_page',
        'dashicons-database-import',
        58
    );
});

function agr_admin_dashboard_page() {
    if (!current_user_can('manage_options')) return;

    // Enregistrer
    if (isset($_POST['agr_save_settings']) && check_admin_referer('agr_save_settings_nonce_action')) {

        $new = [
            'import_clients'   => !empty($_POST['import_clients']) ? 1 : 0,
            'import_contrats'  => !empty($_POST['import_contrats']) ? 1 : 0,
            'import_cotations' => !empty($_POST['import_cotations']) ? 1 : 0,
            'enable_history'   => !empty($_POST['enable_history']) ? 1 : 0,
            'disable_heavy_hooks' => !empty($_POST['disable_heavy_hooks']) ? 1 : 0,
        ];
        agr_update_settings($new);
        echo '<div class="updated"><p>✅ Paramètres enregistrés.</p></div>';
    }

    $s = agr_get_settings();
    ?>
    <div class="wrap">
        <h1>Agrégateur — Tableau de bord</h1>
        <form method="post">
           <?php wp_nonce_field('agr_save_settings_nonce_action'); ?>


            <h2>Contenu à importer</h2>
            <label><input type="checkbox" name="import_clients" value="1" <?php checked(1, $s['import_clients']); ?>> Importer les clients</label><br>
            <label><input type="checkbox" name="import_contrats" value="1" <?php checked(1, $s['import_contrats']); ?>> Importer les contrats</label><br>
            <label><input type="checkbox" name="import_cotations" value="1" <?php checked(1, $s['import_cotations']); ?>> Importer les cotations</label><br>

            <h2>Options</h2>
            <label><input type="checkbox" name="enable_history" value="1" <?php checked(1, $s['enable_history']); ?>> Activer l’historique</label><br>
            <label><input type="checkbox" name="disable_heavy_hooks" value="1" <?php checked(1, $s['disable_heavy_hooks']); ?>> Désactiver les hooks lourds pendant l’import (recommandé)</label><br>

            <p style="margin-top:16px;">
                <button type="submit" name="agr_save_settings" class="button button-primary">Enregistrer</button>
            </p>
        </form>

        <hr>

        <h2>Commandes rapides</h2>
        <p>
            <a class="button" href="<?php echo esc_url(home_url('/?agr=start')); ?>">Démarrer la campagne CRON (scan inbox)</a>
            <a class="button" href="<?php echo esc_url(home_url('/?agr=next')); ?>">Traiter le prochain fichier</a>
            <a class="button button-secondary" href="<?php echo esc_url(home_url('/?agr=run')); ?>">Start + Next</a>
        </p>
    </div>
    <?php
}

/* ========= IMPORT réutilisable (form + cron) ========= */
function agr_import_xml_file($file, $compagnie, $is_front = false) {


    _agr_remove_new_labels();

    $xml = @simplexml_load_file($file);
    if (!$xml) {
        error_log("[AGRÉGATEUR] ❌ Impossible de lire le fichier XML : " . $file);
        return [
            'success' => false,
            'message' => 'Erreur XML'
        ];
    }

    error_log("[AGRÉGATEUR] ✅ XML chargé | compagnie=" . $compagnie);

    $settings = agr_get_settings();
    $profile  = agr_company_profile($compagnie);

    // Option: couper les hooks lourds pendant import
    $hooks_disabled = false;
    if (!empty($settings['disable_heavy_hooks'])) {
        remove_action('save_post', 'get_total_contrats_client', 10, 3);
        $hooks_disabled = true;
    }


        // --- Récupérer les codes conseillers autorisés depuis le répéteur d'options ACF 'codes_conseillers'
        $codes_autorises = [];
        if (function_exists('get_field')) {
            $codes_repeater = get_field('codes_conseillers', 'option');
            if ($codes_repeater && is_array($codes_repeater)) {
                foreach ($codes_repeater as $row) {
                    $code = trim((string)($row['code_conseiller'] ?? ''));
                    if ($code !== '') $codes_autorises[] = $code;
                }
            }
        }
        // Normaliser en chaînes
        $codes_autorises = array_map('strval', $codes_autorises);
        error_log("[AGRÉGATEUR] Codes conseillers autorisés : " . (empty($codes_autorises) ? '(aucun filtre - tout autorisé)' : implode(', ', $codes_autorises)));

        $KEYS = [
            'prenom_client_ag' => 'field_661e36487d613',
            'nom_client_ag'    => 'field_661e364876009',
            'date_naissance_client_ag' => 'field_661e3648724f0',
            'xnom_de_naissance'=> 'field_661e36486ea4e',
            'xrue_perso'       => 'field_661e3648c6fa3',
            'xcode_postal_perso'=> 'field_661e3648ce51d',
            'xville_perso'     => 'field_661e3648d5e49',

            'numero_contrat'   => 'field_690ca019c6d22',
            'code_etat_contrat'=> 'field_690cae04d0fbc',
            'ref_id_personne'  => 'field_690cab88d0fab',
            'ref_id_produit'   => 'field_690caac4d0fa2',
            'libelle_produit'  => 'field_690caad1d0fa3',
            'ref_id_conseiller'=> 'field_690caae5d0fa4',
            'libelle_conseiller'=> 'field_690caaf3d0fa5',
            'date_deffet_contrat'=> 'field_690cab36d0fa6',
            'sommes_versees'   => 'field_690cab5fd0fa8',
            'valeur_rachat'    => 'field_690cab4ad0fa7',
            'date_situation'   => 'field_690cab6ad0fa9',

            'actions_repeater' => 'field_690cab7bd0faa',
            'action_ref_id_support' => 'field_690caba2d0fac',
            'action_libelle_support' => 'field_690cabbcd0fad',
            'action_valeur_cotation_euro' => 'field_690cabd6d0fae',
            'action_nombre_de_parts' => 'field_690cabe4d0faf',
            'action_montant_en_euro' => 'field_690cabeed0fb0',
            'action_indicateur_pmv' => 'field_690cabf8d0fb1',
            'action_plus_value_euro' => 'field_690cadc0d0fb9',
            'action_plus_value_pct' => 'field_690cadd7d0fba',
            'action_pourcentage_des_actions' => 'field_690cad8bd0fb7',

            'evaluation_contrat' => 'field_690cad52d0fb4',
            'evaluation_totale'  => 'field_690cad46d0fb3',
            'pourcentage_du_contrat' => 'field_690cad6bd0fb5',

            'action_code_isin' => 'field_690dc71a0f2ee',
            'plus_value_totale_en_euros' => 'field_690dc590bd253',
            'plus_value_totale_en_pourcentage' => 'field_690dc5a0bd254',

            'etablissement' => 'field_690cacfbd0fb2',

            // 🆕 Ajout des nouveaux champs
            'nom_client_contrat' => 'field_690dedd6cd83b',
            'prenom_client_contrat' => 'field_690dedddcd83c',
        ];

        $counters = ['clients_new'=>0,'clients_upd'=>0,'clients_skip'=>0,'contrats_new'=>0,'contrats_upd'=>0];
        $totaux = [];

        _agr_ensure_category('Clients');
        _agr_ensure_category('Agrégateur');
        _agr_ensure_category('Contrats');
        _agr_ensure_category('Nouveau client');
        _agr_ensure_category('Nouveau contrat');

        /* =========================================================
         *  PRÉ-CALCUL : construire la liste des ID_personne autorisés
         *  -> on parcourt les contrats et si ref_ID_conseiller est autorisé (ou pas de filtre),
         *     on récupère le role/ref_ID_personne et on l'ajoute.
         * ========================================================= */
        $allowed_person_ids = []; // liste des ID_personne autorisés (strings)
        $contrat_nodes_all = $xml->xpath("//donnee_reguliere//destinataire//contrats//contrat") ?: [];
        if (!empty($contrat_nodes_all)) {
            foreach ($contrat_nodes_all as $cnode) {
                $ref_conseiller_c = (string)$cnode['ref_ID_conseiller'];
                // si on a un filtre et que ce ref_conseiller n'est pas dans la liste, on ignore pour la collecte
                if (!empty($codes_autorises) && $ref_conseiller_c !== '' && !in_array((string)$ref_conseiller_c, $codes_autorises, true)) {
                    continue;
                }
                // récupérer le role/ref_ID_personne
                $role_node = $cnode->xpath(".//role")[0] ?? null;
                if ($role_node) {
                    $ref_personne_role = (string)$role_node['ref_ID_personne'];
                    if ($ref_personne_role !== '') {
                        $allowed_person_ids[$ref_personne_role] = true;
                    }
                }
            }
        }
        // si $codes_autorises est vide => autoriser tout le monde (vide signifie aucun filtre)
        $filter_by_codes = !empty($codes_autorises);
        if ($filter_by_codes) {
            error_log("[AGRÉGATEUR] Filtrage activé. Nombre ID_personne extraits depuis contrats autorisés: " . count($allowed_person_ids));
        } else {
            error_log("[AGRÉGATEUR] Pas de filtrage par codes (import complet).");
        }

        /* =========================================================
         *  CLIENTS (création / mise à jour) — on filtre si besoin
         * ========================================================= */
        if (!empty($settings['import_clients'])) {

        $persons = $xml->xpath("//donnee_reguliere//destinataire//personnes//personne");
        if (!$persons) {
            error_log("[AGRÉGATEUR] ⚠️ Aucun noeud <personne> trouvé");
        } else {
            error_log("[AGRÉGATEUR] ✅ ".count($persons)." personnes trouvées");
        }

        foreach ($persons as $p) {
            $nom = (string)$p['nom_d_usage'];
            $prenom = (string)$p['prenoms'];
            $id_personne = (string)$p['ID_personne'] ?? '';

            // Si on filtre, skip si cet ID_personne n'est pas dans la liste autorisée
            if ($filter_by_codes && $id_personne !== '' && !isset($allowed_person_ids[$id_personne])) {
                $counters['clients_skip']++;
                continue;
            }

            if (!$nom || !$prenom) continue;

            $title = $nom . ' ' . $prenom;
            error_log("[AGRÉGATEUR][CLIENT] $title (ID_personne: {$id_personne})");

            // Vérifier si le client existe déjà
            $existing = get_posts([
                'post_type'=>'clients',
                'posts_per_page'=>1,
                'meta_query'=>[
                    ['key'=>'nom_client_ag','value'=>$nom],
                    ['key'=>'prenom_client_ag','value'=>$prenom]
                ]
            ]);
            $addr_attr = $profile['person_addr_attr'] ?? 'adr_ligne1';
            $fields = [
                $KEYS['nom_client_ag'] => $nom,
                $KEYS['prenom_client_ag'] => $prenom,
                $KEYS['date_naissance_client_ag'] => format_date_fr((string)$p['date_naissance']),
                $KEYS['xnom_de_naissance'] => (string)$p['nom_patronymique'],
                $KEYS['xrue_perso'] => (string)$p[$addr_attr],
                $KEYS['xville_perso'] => (string)$p['ville'],
                $KEYS['xcode_postal_perso'] => (string)$p['code_postal']
            ];

            $cat_ids = [
                _agr_ensure_category('Clients'),
                _agr_ensure_category('Agrégateur'),
            ];
            if ($compagnie) $cat_ids[] = _agr_ensure_category($compagnie);

            // --- Déterminer le post ID (existant ou nouveau)
            if ($existing) {
                $pid = $existing[0]->ID;
                foreach ($fields as $k=>$v) update_field($k,$v,$pid);
                @wp_remove_object_terms($pid, 'Nouveau client', 'category');
                wp_set_post_categories($pid, array_filter($cat_ids), false);
                $counters['clients_upd']++;
} else {
    $pid = wp_insert_post([
        'post_title'=>$title,
        'post_type'=>'clients',
        'post_status'=>'publish'
    ]);

    // Mise à jour champs
    foreach ($fields as $k=>$v) update_field($k,$v,$pid);

    /* =====================================================
       🔵 AJOUT AUTOMATIQUE type_de_contact = Client
       ===================================================== */
    if (agr_should_log_history($compagnie)) {

        update_field('type_de_contact', ['Client'], $pid);
    }

/* =====================================================
   🔵 HISTORIQUE : Nouveau client importé
   ===================================================== */
if (agr_should_log_history($compagnie)) {

    $client_title = $title;

    $message =
        '<span style="margin-right:12px;">'
        . '💻'
        . '</span>'
        . '<span style="margin-right:8px;">'
        . esc_html($client_title)
        . ' a été importé par l\'agrégation ' . esc_html($compagnie)
        . '</span>';

    $historique = get_field('historique_rep', 'option') ?: [];
    $historique[] = [
        'historique_sous' => $message,
        'timestamp'       => current_time('Y-m-d H:i:s'),
    ];
    update_field('historique_rep', $historique, 'option');
}
/* ===================================================== */


    // Catégorie
    $cat_ids[] = _agr_ensure_category('Nouveau client');
    wp_set_post_categories($pid, array_filter($cat_ids), false);

    $counters['clients_new']++;
}



            // --- Ajouter ref_personne dans le répéteur nums_client
            $ref_personne = $id_personne;
            if ($ref_personne) {
                $existing_nums = get_field('nums_client', $pid);
                if (!is_array($existing_nums)) $existing_nums = [];

                // Eviter doublons
                $already_exists = false;
                foreach ($existing_nums as $row) {
                    if (($row['numeros_client'] ?? '') === $ref_personne) {
                        $already_exists = true;
                        break;
                    }
                }

                if (!$already_exists) {
                    $existing_nums[] = ['numeros_client' => $ref_personne];
                    update_field('nums_client', $existing_nums, $pid);
                }
            }
        } // foreach persons
/* =====================================================
   🔵 HISTORIQUE : Recap des clients importés
   ===================================================== */
if (agr_should_log_history($compagnie) && $counters['clients_new'] > 0) {

    $nb = $counters['clients_new'];

    // Phrase en fonction du singulier / pluriel
    $texte = ($nb === 1)
        ? "1 client a été importé par l'agrégation {$compagnie}"
        : "$nb clients ont été importés par l'agrégation {$compagnie}";

    $message =
        '<span style="margin-right:12px;">'
        . '💻'
        . '</span>'
        . '<span style="margin-right:8px;">' . $texte . '</span>';

    $historique = get_field('historique_rep', 'option') ?: [];
    $historique[] = [
        'historique_sous' => $message,
        'timestamp'       => current_time('Y-m-d H:i:s'),
    ];
    update_field('historique_rep', $historique, 'option');
}
}


        /* =========================================================
         *  CONTRATS (création / mise à jour)
         *  -> ici on applique le filtre sur ref_ID_conseiller si $codes_autorises non vide
         * ========================================================= */

        if (!empty($settings['import_contrats'])) {

        $contrats = $xml->xpath("//donnee_reguliere//destinataire//contrats//contrat");
        if (!$contrats) error_log("[AGRÉGATEUR] ⚠️ Aucun contrat trouvé");

        foreach ($contrats as $contrat) {
            $ref_conseiller = (string)$contrat['ref_ID_conseiller'];
            // Si on filtre et que ce contrat n'appartient pas à un conseiller autorisé -> skip
            if ($filter_by_codes && $ref_conseiller !== '' && !in_array((string)$ref_conseiller, $codes_autorises, true)) {
                continue;
            }

            $num = (string)$contrat['num_contrat'] ?: (string)$contrat['numero_contrat'] ?: (string)$contrat['num'];
            if (!$num) continue;

            // Récupérer la liste des ISIN UCS depuis la page d'options
            $ucs_isins = (function_exists('get_field') ? get_field('contrats_ucs', 'option') : []) ?: [];
            $ucs_isins_list = [];
            foreach ($ucs_isins as $u) {
                if (!empty($u['isin_ucs'])) $ucs_isins_list[] = $u['isin_ucs'];
            }

            // Vérification si ce contrat contient un ISIN UCS
            $supports = $contrat->xpath(".//support_composants/support_composant");
            $has_ucs = false;
            if ($supports) {
                foreach ($supports as $s) {
                    $profile = $profile ?? agr_company_profile($compagnie);

$ref_support = (string)$s['ref_ID_support_info_generale'];

// Si Abeille: ref_support EST l’ISIN
if (!empty($profile['support_id_is_isin'])) {
    $isin = $ref_support;
} else {
    $support_node = $xml->xpath("//support_info_gen_consomme[@{$profile['support_lookup_attr']}='{$ref_support}']")[0] ?? null;
    $isin = $support_node ? (string)$support_node[$profile['support_isin_attr']] : '';
}
                    if ($isin && in_array($isin, $ucs_isins_list)) {
                        $has_ucs = true;
                        break;
                    }
                }
            }

            // Déterminer le CPT cible
            $cpt_target = $has_ucs ? 'contrats-avec-ucs' : 'contrats';

            // Vérifier si le contrat existe déjà dans le CPT cible
            $existing_contract = get_page_by_title($num, OBJECT, $cpt_target);

            if ($existing_contract && $existing_contract instanceof WP_Post && $existing_contract->post_status !== 'trash') {
                $post_id = $existing_contract->ID;
                $is_new_contract = false;
                $counters['contrats_upd']++;
} else {
    $post_id = wp_insert_post([
        'post_title' => $num,
        'post_type' => $cpt_target,
        'post_status' => 'publish'
    ]);
    $is_new_contract = true;
    $counters['contrats_new']++;

    
    /* ===================================================== */
}



            update_field($KEYS['numero_contrat'], $num, $post_id);
            update_field($KEYS['code_etat_contrat'], (string)$contrat['code_etat_contrat'], $post_id);
            update_field($KEYS['etablissement'], $compagnie, $post_id);

            // 🆕 Nom et prénom du client pour chaque contrat
            $role_node = $contrat->xpath(".//role")[0] ?? null;
            $ref_personne_role = $role_node ? (string)$role_node['ref_ID_personne'] : '';
            update_field($KEYS['ref_id_personne'], $ref_personne_role, $post_id);

            $personne_node = $xml->xpath("//personne[@ID_personne='{$ref_personne_role}']")[0] ?? null;

/* =====================================================
   🔵 HISTORIQUE : Nouveau contrat importé
   ===================================================== */
if ($is_new_contract && agr_should_log_history($compagnie)) {

    // On récupère nom/prénom du client
    $client_nom = (string)($personne_node['nom_d_usage'] ?? '');
    $client_prenom = (string)($personne_node['prenoms'] ?? '');
    $client_title = trim($client_nom . ' ' . $client_prenom);

    $message =
        '<span style="margin-right:12px;">'
        . '💻'
        . '</span>'
        . '<span style="margin-right:8px;">'
        . esc_html($client_title)
        . ' a reçu un nouveau contrat par l\'agrégation ' . esc_html($compagnie)
        . '</span>';

    $historique = get_field('historique_rep', 'option') ?: [];
    $historique[] = [
        'historique_sous' => $message,
        'timestamp'       => current_time('Y-m-d H:i:s'),
    ];
    update_field('historique_rep', $historique, 'option');
}
/* ===================================================== */



            if ($personne_node) {
                update_field($KEYS['nom_client_contrat'], (string)$personne_node['nom_d_usage'], $post_id);
                update_field($KEYS['prenom_client_contrat'], (string)$personne_node['prenoms'], $post_id);
            }

            $ref_produit = (string)$contrat['ref_ID_produit'];
            update_field($KEYS['ref_id_produit'], $ref_produit, $post_id);
            $prod = $xml->xpath("//donnee_specifique//produit[@ID_produit='{$ref_produit}']")[0] ?? null;
            $libelle_produit = $prod ? (string)$prod['designation'] : '';
            update_field($KEYS['libelle_produit'], $libelle_produit, $post_id);

            $ref_conseiller = (string)$contrat['ref_ID_conseiller'];
            update_field($KEYS['ref_id_conseiller'], $ref_conseiller, $post_id);
            $inter = $xml->xpath("//intermediaire_consomme//intermediaire[@ID_intermediaire='{$ref_conseiller}']")[0] ?? null;
            $libelle_conseiller = $inter ? (string)$inter['nom'] : '';
            update_field($KEYS['libelle_conseiller'], $libelle_conseiller, $post_id);

            // Calcul plus précis des sommes versées
            $supports_calc = $contrat->xpath(".//support_composants/support_composant");
            $total_sommes_calc = 0;
            if ($supports_calc) {
                foreach ($supports_calc as $sup) {
                    $ind = parse_number((string)$sup['indicateur_de_PMV']);
                    $parts = parse_number((string)$sup['nombre_de_parts']);
                    $total_sommes_calc += ($ind * $parts);
                }
            }
            update_field($KEYS['sommes_versees'], format_euro($total_sommes_calc), $post_id);

            $valeur_rachat = parse_number((string)$contrat['valeur_rachat']);
            update_field($KEYS['valeur_rachat'], format_euro($valeur_rachat), $post_id);
            $date_effet = (string)$contrat['date_effet_contrat'];
            update_field($KEYS['date_deffet_contrat'], format_date_fr($date_effet), $post_id);
            $date_situation = (string)$contrat['date_situation'];
            update_field($KEYS['date_situation'], format_date_fr($date_situation), $post_id);

            /* ========= Répéteur actions ========= */
$supports = $contrat->xpath(".//support_composants/support_composant");
$rows = [];
$total_montant = 0;
$total_plus_euros = 0;
$total_base_pmv = 0;
$raw_actions = [];

if ($supports) {
    foreach ($supports as $s) {
        $profile = $profile ?? agr_company_profile($compagnie);

        $ref_support = (string)$s['ref_ID_support_info_generale'];
        $val_cot   = parse_number((string)$s['valeur_cotation_euro']);
        $nb_parts  = parse_number((string)$s['nombre_de_parts']);
        $montant   = parse_number((string)$s['montant_en_euro']);
        $ind_pmv   = parse_number((string)$s['indicateur_de_PMV']);

        $libelle_support = '';
        $code_isin = '';

        // Abeille: ref_support = ISIN direct
        if (!empty($profile['support_id_is_isin'])) {
            $code_isin = $ref_support;

            $lib_node = $xml->xpath("//support_info_gen_consomme[@{$profile['support_lookup_attr']}='{$ref_support}']")[0] ?? null;
            if ($lib_node && isset($lib_node[$profile['support_label_attr']])) {
                $libelle_support = (string)$lib_node[$profile['support_label_attr']];
            }
        } else {
            $lib_node = $xml->xpath("//support_info_gen_consomme[@{$profile['support_lookup_attr']}='{$ref_support}']")[0] ?? null;
            if ($lib_node) {
                if (isset($lib_node[$profile['support_label_attr']])) $libelle_support = (string)$lib_node[$profile['support_label_attr']];
                if (isset($lib_node[$profile['support_isin_attr']]))  $code_isin = (string)$lib_node[$profile['support_isin_attr']];
            }
        }

        $plus_euro = ($val_cot * $nb_parts) - ($ind_pmv * $nb_parts);
        $plus_pct  = ($val_cot * $nb_parts) > 0 ? ($plus_euro / ($val_cot * $nb_parts)) * 100 : 0;

        $total_montant     += $montant;
        $total_plus_euros  += $plus_euro;
        $total_base_pmv    += ($ind_pmv * $nb_parts);

        $raw_actions[] = [
            'ref_support'     => $ref_support,
            'libelle_support' => $libelle_support,
            'code_isin'       => $code_isin,
            'val_cot'         => $val_cot,
            'nb_parts'        => $nb_parts,
            'montant'         => $montant,
            'ind_pmv'         => $ind_pmv,
            'plus_euro'       => $plus_euro,
            'plus_pct'        => $plus_pct
        ];
    }

    foreach ($raw_actions as $ra) {
        $pct_action = $total_montant > 0 ? ($ra['montant'] * 100 / $total_montant) : 0;
        $rows[] = [
            $KEYS['action_ref_id_support'] => $ra['ref_support'],
            $KEYS['action_libelle_support'] => $ra['libelle_support'],
            $KEYS['action_code_isin'] => $ra['code_isin'],
            $KEYS['action_valeur_cotation_euro'] => format_euro($ra['val_cot']),
            $KEYS['action_nombre_de_parts'] => $ra['nb_parts'],
            $KEYS['action_montant_en_euro'] => format_euro($ra['montant']),
            $KEYS['action_indicateur_pmv'] => $ra['ind_pmv'],
            $KEYS['action_plus_value_euro'] => format_euro($ra['plus_euro']),
            $KEYS['action_plus_value_pct'] => format_pct($ra['plus_pct']),
            $KEYS['action_pourcentage_des_actions'] => format_pct($pct_action)
        ];
    }
}



            

            update_field($KEYS['actions_repeater'], $rows, $post_id);
            update_field($KEYS['evaluation_contrat'], format_euro($total_montant), $post_id);
            update_field($KEYS['plus_value_totale_en_euros'], format_euro($total_plus_euros), $post_id);
            $pct_tot = $total_base_pmv > 0 ? ($total_plus_euros / $total_base_pmv) * 100 : 0;
            update_field($KEYS['plus_value_totale_en_pourcentage'], format_pct($pct_tot), $post_id);

            $cat_ids_contract = [
                _agr_ensure_category('Contrats'),
                _agr_ensure_category('Agrégateur'),
            ];

            if ($libelle_produit) $cat_ids_contract[] = _agr_ensure_category($libelle_produit);

            // ✅ Ajout de la catégorie UCS si CPT = contrats-avec-ucs
            if ($cpt_target === 'contrats-avec-ucs') {
                $cat_ids_contract[] = _agr_ensure_category('UCS');
            }

            if ($is_new_contract) {
                $cat_ids_contract[] = _agr_ensure_category('Nouveau contrat');
                wp_set_post_categories($post_id, array_filter($cat_ids_contract), false);
            } else {
                @wp_remove_object_terms($post_id, 'Nouveau contrat', 'category');
                wp_set_post_categories($post_id, array_filter($cat_ids_contract), false);
            }

            if (!isset($totaux[$ref_personne_role])) $totaux[$ref_personne_role] = 0;
            $totaux[$ref_personne_role] += $total_montant;
        } // foreach contrats

/* =====================================================
   🔵 HISTORIQUE : Recap des contrats importés
   ===================================================== */
if (agr_should_log_history($compagnie) && $counters['contrats_new'] > 0) {

    $nb = $counters['contrats_new'];

    // Phrase en fonction du singulier / pluriel
    $texte = ($nb === 1)
        ? "1 contrat a été importé par l'agrégation {$compagnie}"
        : "$nb contrats ont été importés par l'agrégation {$compagnie}";

    $message =
        '<span style="margin-right:12px;">'
        . '💻'
        . '</span>'
        . '<span style="margin-right:8px;">' . $texte . '</span>';

    $historique = get_field('historique_rep', 'option') ?: [];
    $historique[] = [
        'historique_sous' => $message,
        'timestamp'       => current_time('Y-m-d H:i:s'),
    ];
    update_field('historique_rep', $historique, 'option');
}

}


        // Mise à jour des totaux pour chaque personne
        foreach ($totaux as $ref=>$total) {
            $contrats_list = get_posts(['post_type'=>'contrats','meta_key'=>'ref_ID_personne','meta_value'=>$ref,'posts_per_page'=>-1]);
            foreach ($contrats_list as $c) {
                $eval = parse_number(get_field($KEYS['evaluation_contrat'],$c->ID));
                $pct = $total > 0 ? ($eval / $total)*100 : 0;
                update_field($KEYS['evaluation_totale'], format_euro($total), $c->ID);
                update_field($KEYS['pourcentage_du_contrat'], format_pct($pct), $c->ID);
            }
        }

// Crée/MAJ un post Cotations pour la date du jour
if (!empty($settings['import_cotations'])) {

    $post_title = date('d/m/Y');

    // Lookup fiable par titre + post_type
    $cotation_post = get_page_by_title($post_title, OBJECT, 'cotations');

    if ($cotation_post instanceof WP_Post) {
        $post_id = $cotation_post->ID;
    } else {
        $post_id = wp_insert_post([
            'post_title'  => $post_title,
            'post_type'   => 'cotations',
            'post_status' => 'publish',
        ]);
    }

    $existing_codes = [];
    $supports = $xml->xpath("//support_composants/support_composant");
    $rows = [];

    if ($supports) {
        foreach ($supports as $s) {
            $valeur = (string)$s['valeur_cotation_euro'];

            $date_raw = (string)$s['date_cotation'];
            $date_parts = explode('-', $date_raw);
            $date_cot = (count($date_parts) === 3)
                ? ($date_parts[2].'/'.$date_parts[1].'/'.$date_parts[0])
                : $date_raw;

            $code = (string)$s['ref_ID_support_info_generale'];

            if (!empty($profile['support_id_is_isin'])) {
                $isin = $code;
                $support_node = $xml->xpath("//support_info_gen_consomme[@{$profile['support_lookup_attr']}='{$code}']")[0] ?? null;
                $libelle = $support_node ? (string)$support_node[$profile['support_label_attr']] : '';
            } else {
                $support_node = $xml->xpath("//support_info_gen_consomme[@{$profile['support_lookup_attr']}='{$code}']")[0] ?? null;
                $isin = $support_node ? (string)$support_node[$profile['support_isin_attr']] : '';
                $libelle = $support_node ? (string)$support_node[$profile['support_label_attr']] : '';
            }

            if ($code && $isin && !isset($existing_codes[$code.'-'.$isin])) {
                $rows[] = [
                    'date'                 => $date_cot,
                    'code'                 => $code,
                    'isin'                 => $isin,
                    'valeur_cotation_euro' => $valeur,
                    'libelle'              => $libelle
                ];
                $existing_codes[$code.'-'.$isin] = true;
            }
        }
    }

    update_field('cotations', $rows, $post_id);

    error_log("[AGRÉGATEUR] Cotations rows (count): " . count($rows));
    error_log(print_r(array_slice($rows, 0, 10), true));
}

// ✅ Affichage UNIQUEMENT quand on est dans le shortcode (front/admin UI)
if ($is_front) {
    echo "<div style='padding:10px;background:#f7f7f7;border:1px solid #ccc;'>
            Clients créés: {$counters['clients_new']} / MAJ: {$counters['clients_upd']} / SKIP (filtre): {$counters['clients_skip']}<br>
            Contrats créés: {$counters['contrats_new']} / MAJ: {$counters['contrats_upd']}
          </div>";
}

if ($hooks_disabled) {
    // ⚠️ si ta fonction n’accepte pas 3 args, mets 1
    add_action('save_post', 'get_total_contrats_client', 10, 3);
}

return [
    'success'  => true,
    'counters' => $counters
];
}


function import_xml_agregateur_shortcode() {
    ob_start();
    error_log("[AGRÉGATEUR] Shortcode lancé (v1.6 + Cotations) - démarrage");

    // Nettoyage labels "Nouveau"
    _agr_remove_new_labels();

    $compagnie = isset($_POST['compagnie']) ? sanitize_text_field($_POST['compagnie']) : '';

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_FILES['xml_file']['tmp_name'])) {
        // ✅ Appel avec $is_front = true pour autoriser l’affichage dans la fonction
        $res = agr_import_xml_file($_FILES['xml_file']['tmp_name'], $compagnie, true);

        if (empty($res['success'])) {
            echo "<p>Erreur XML</p>";
            return ob_get_clean();
        }

        // ⚠️ IMPORTANT :
        // Ne ré-echo pas le bloc de compteurs ici,
        // car agr_import_xml_file(..., true) l’affiche déjà.
    }
    ?>
    <div style="display:flex;gap:20px;flex-wrap:wrap;">
        <form method='post' enctype='multipart/form-data' style='margin-top:20px;min-width:280px;'>
            <h3>Import SwissLife</h3>
            <input type='hidden' name='compagnie' value='SwissLife'>
            <label><strong>Importer un fichier XML :</strong></label><br>
            <input type='file' name='xml_file' accept='.xml' required style='margin-top:8px;'><br><br>
            <button type='submit' style='background:#2271b1;color:#fff;border:none;padding:8px 16px;border-radius:4px;cursor:pointer;'>
                Importer SwissLife
            </button>
        </form>
    </div>
    <?php

    return ob_get_clean();
}

?>
