<?php

add_action('init', function() { 
    add_shortcode('liste_des_ventes', 'render_liste_des_ventes'); 
});

function render_liste_des_ventes() {
    if (!function_exists('get_field')) return '<p>ACF requis.</p>';

// 🔵 Vérification du rôle
$current_user = wp_get_current_user();
$roles_autorises = ['administrator', 'code-maitre']; // <-- ton rôle personnalisé
$can_edit = array_intersect($roles_autorises, (array) $current_user->roles);
$can_edit = !empty($can_edit);

    // Sauvegarde des modifications
    if ($can_edit && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_ventes'])) {

        foreach ($_POST['ventes'] as $id => $vals) {
            $id = intval($id);
            foreach ($vals as $k => $v) {
                update_field($k, sanitize_text_field($v), $id);
            }
        }
        echo '<script>window.location.href = window.location.href;</script>';
        return;
    }

    // Récupération des ventes
    $ventes = get_posts([
        'post_type'      => 'post',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'category_name'  => 'nouvelle-vente',
    ]);

$ventes = array_filter($ventes, function($vente) use ($can_edit, $current_user) {

    // 1) Toujours garder seulement les ventes qui ont un client
    $client = get_field('prenom_du_client', $vente->ID) ?: get_field('nom_du_client', $vente->ID);
    if (empty($client)) return false;

    // 2) Si admin / code-maitre => voit tout
    if ($can_edit) return true;

    // 3) Sinon => ne voir que les ventes où il est impliqué
    $uid = (int) ($current_user->ID ?? 0);
    if (!$uid) return false;

    $fields = [
        get_field('mandataire', $vente->ID),
        get_field('accompagnant', $vente->ID),
        get_field('second_accompagnant', $vente->ID),
    ];

    // helper pour extraire un ID user depuis ACF (ID / array / WP_User)
    $extract_id = function($val) {
        if (is_numeric($val)) return (int)$val;
        if (is_array($val)) {
            if (!empty($val['ID']) && is_numeric($val['ID'])) return (int)$val['ID'];
            if (!empty($val['id']) && is_numeric($val['id'])) return (int)$val['id'];
        }
        if (is_object($val) && isset($val->ID) && is_numeric($val->ID)) return (int)$val->ID;
        return 0;
    };

    foreach ($fields as $f) {
        if ($extract_id($f) === $uid) return true;
    }

    return false;
});


usort($ventes, function($a, $b) {

    $da = get_field('date_du_jour', $a->ID);
    $db = get_field('date_du_jour', $b->ID);

    $dta = DateTime::createFromFormat('d/m/Y', (string)$da) ?: DateTime::createFromFormat('Y-m-d', (string)$da);
    $dtb = DateTime::createFromFormat('d/m/Y', (string)$db) ?: DateTime::createFromFormat('Y-m-d', (string)$db);

    $ta = $dta ? $dta->getTimestamp() : 0;
    $tb = $dtb ? $dtb->getTimestamp() : 0;

    // DESC = plus récent en premier
    if ($ta === $tb) {
        return strcmp($b->post_date, $a->post_date);
    }
    return $tb <=> $ta;
});

    if (empty($ventes)) return '<p>Aucune vente trouvée.</p>';

    // Collecte des années disponibles via DateTime
    $annees = [];
    foreach($ventes as $vente){
        $date_raw = get_field('date_du_jour', $vente->ID);
        if($date_raw){
            $dt = DateTime::createFromFormat('Y-m-d', $date_raw) ?: DateTime::createFromFormat('d/m/Y', $date_raw);
            if($dt) $annees[] = $dt->format('Y');
        }
    }
    $annees = array_unique($annees);
    rsort($annees);

    ob_start(); ?>

<style>
/* ---------- TABLE ---------- */
.liste-ventes-table {
  width: calc(100% - 40px);
  margin: 20px;
  border-collapse: collapse;
  background: #fff;
  border: 1px solid #ccc;
  font-size: 14px;
  line-height: 24px;
}
.liste-ventes-table th {
  background-color: #2271b1;
  color: #fff;
  font-weight: 600;
  text-transform: uppercase;
  font-size: 16px;
  text-align: center;
  padding: 10px;
  border: 1px solid #ccc;
}
.liste-ventes-table td {
  border: 1px solid #eee;
  text-align: center;
  padding: 0 10px;
  height: 28px;
  vertical-align: middle;
  font-size: 14px;
}

/* ---------- INPUTS + SELECTS ---------- */
.liste-ventes-table td input[type="number"],
.liste-ventes-table td select {
  font-size: 14px !important;
  border: none !important;
  border-radius: 0 !important;
  width: 100%;
  height: 28px;
  background: transparent !important;
  text-align: center;
  line-height: 28px !important;
  box-shadow: none !important;
  margin: 0 !important;
}

/* ---------- SELECT avec flèche SVG ---------- */
.liste-ventes-table td select {
  appearance: none;
  -webkit-appearance: none;
  -moz-appearance: none;
  position: relative;
  padding-right: 30px !important;
  cursor: pointer;
  background-image: url("data:image/svg+xml,%3Csvg width='24' height='24' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M7 10l5 5 5-5H7z' fill='%232271b1'/%3E%3C/svg%3E");
  background-repeat: no-repeat;
  background-position: right 8px center;
  background-size: 18px;
}

/* Options menu ouvertes */
.liste-ventes-table td select option {
  text-align: left;
}

/* Focus doux */
.liste-ventes-table td select:focus,
.liste-ventes-table td input:focus {
  outline: none;
  background: #eef6fc;
}

/* Champs éditables */
.liste-ventes-table td.editable input {
  background: #fff;
  transition: background 0.2s ease;
}
.liste-ventes-table td.editable input:focus {
  background: #eef6fc;
  border-bottom: 1px solid #2271b1;
}

/* ---------- POUBELLE ---------- */
.delete-icon {
  color: #d00;
  font-size: 18px;
  cursor: pointer;
  margin: 0 5px;
  transition: transform 0.2s ease, color 0.2s ease;
}
.delete-icon:hover {
  color: #a00;
  transform: scale(1.2);
}

/* ---------- INFOBULLE ---------- */
.tooltip-container {
  position: relative;
  display: inline-block;
}
.tooltip-icon {
  color: #2271b1;
  cursor: pointer;
  font-size: 16px;
  vertical-align: middle;
}
.tooltip-box {
  display: none;
  position: absolute;
  top: 28px;
  left: 50%;
  transform: translateX(-50%);
  background: #fff;
  color: #333;
  border: 1px solid #ccc;
  border-radius: 6px;
  padding: 8px 10px;
  width: 220px;
  box-shadow: 0 2px 8px rgba(0,0,0,0.15);
  font-size: 13px;
  line-height: 1.4;
  z-index: 9999;
}
.tooltip-container:hover .tooltip-box {
  display: block;
}

/* ---------- BOUTON ---------- */
.button-validate {
  margin-top: 15px;
  padding: 10px 15px;
  background-color: #2271b1;
  border: none;
  border-radius: 4px;
  color: #fff;
  font-weight: 600;
  cursor: pointer;
  font-size: 14px;
  transition: background-color 0.2s ease;
}
.button-validate:hover {
  background-color: #1c5f91;
}

/* ---------- DROPDOWN ANNEE ---------- */
.ventes-filter-wrapper {
    margin-top: 0px; 
    margin-bottom: 20px;
    display: inline-flex;
    flex-direction: column;
    gap: 6px;
}
.ventes-filter-label {
    font-size: 14px;
    font-weight: 700;
    letter-spacing: 0.5px;
    color: #333;
}
.ventes-filter-select {
    width: 100px !important;
    padding: 6px 10px !important;
    font-size: 13px;
    border-radius: 6px;
    border: 1px solid #ccc;
    background: #fff;
    cursor: pointer;
    outline: none;
    display: inline-block;
    height: 35px !important;
    font-size: 14px !important;
    border-radius: 6px !important;
}
.ventes-filter-select:hover {
    border-color: #999;
}
.ventes-filter-select:focus {
    border-color: #0073aa;
    box-shadow: 0 0 0 1px #0073aa44;
}

/* ---------- PAGINATION ---------- */
.pagination-wrapper {display:flex; justify-content:center; align-items:center; margin:15px 0 30px; gap:6px; font-family: Arial, sans-serif;}
.pagination-wrapper span {display:inline-block; padding:6px 10px; background-color:#f0f0f0; color:#333; cursor:pointer; border-radius:4px; font-weight:500; user-select:none; transition: all 0.2s ease;}
.pagination-wrapper span.active {background-color:#2271b1; color:#fff; font-weight:700;}
.pagination-wrapper span:hover:not(.active){background-color:#d1e4fc;}
.pagination-wrapper span.disabled{background-color:#eee;color:#aaa;cursor:default;}
</style>

<div class="ventes-filter-wrapper">
    <label for="annee-select" class="ventes-filter-label">ANNÉES</label>
    <select id="annee-select" class="ventes-filter-select">
        <?php
        $current_year = (int) date('Y');
        $years = [$current_year, $current_year - 1];
foreach ($annees as $year) {
    echo '<option value="' . esc_attr($year) . '">' . esc_html($year) . '</option>';
}
        ?>
    </select>
</div>

<form method="post" id="form-liste-ventes">
<table class="liste-ventes-table">
  <thead>
    <tr>
      <th>Statut</th>
      <th>Date</th>
      <th>Client</th>
      <th>Contrat</th>
      <th>Conseiller(s)</th>
      <th>Volume</th>
      <th>Frais</th>
      <th>Commission (€)</th>
      <th>Plus d’infos</th>
      <th></th>
    </tr>
  </thead>
  <tbody id="ventes-tbody">
  <?php foreach ($ventes as $vente):
    $id = $vente->ID;
    $statut = get_field('statut_vente', $id) ?: 'EN COURS';
    $date = get_field('date_du_jour', $id);
    $client = trim(get_field('prenom_du_client', $id).' '.get_field('nom_du_client', $id));
    $contrat_id = get_field('veuillez_selectionner_le_contrat_souscrit', $id);
    $mandataire = get_field('mandataire', $id);
    $acc1 = get_field('accompagnant', $id);
    $acc2 = get_field('second_accompagnant', $id);
    $volume = floatval(get_field('volume_total_signe', $id));
    $sagit_ucs = get_field('sagit_il_dune_ucs_', $id);
    $frais_user = floatval(get_field('frais', $id));

    // Conseillers
   // =======================
// Conseillers (robuste + bruts)
// =======================
$mandataire = get_field('mandataire', $id);
$acc1 = get_field('accompagnant', $id);
$acc2 = get_field('second_accompagnant', $id);

// champs bruts (anciens collaborateurs)
$mandataire_brut = get_field('mandataire_nom_brut', $id);
$acc1_brut       = get_field('accompagnant_nom_brut', $id);
$acc2_brut       = get_field('second_accompagnant_nom_brut', $id);

// Convertit un champ ACF user (ID/array/object/string) en "nom affichable" + bool "existe"
$get_user_display_and_exists = function($val) {
    // ACF peut renvoyer array
    if (is_array($val)) {
        $id = $val['ID'] ?? $val['id'] ?? null;
        $name = $val['display_name'] ?? $val['user_firstname'] ?? '';
        if ($name) return [$name, true];
        if ($id && is_numeric($id)) {
            $u = get_userdata((int)$id);
            if ($u) return [$u->display_name, true];
        }
        return ['', false];
    }

    // ACF peut renvoyer WP_User
    if (is_object($val) && isset($val->ID)) {
        return [($val->display_name ?? ''), true];
    }

    // ID
    if (is_numeric($val)) {
        $u = get_userdata((int)$val);
        if ($u) return [$u->display_name, true];
        return ['', false];
    }

    // string (au cas où)
    $s = trim((string)$val);
    if ($s !== '' && $s !== '-') return [$s, false];

    return ['', false];
};

list($m_name, $m_exists)   = $get_user_display_and_exists($mandataire);
list($a1_name, $a1_exists) = $get_user_display_and_exists($acc1);
list($a2_name, $a2_exists) = $get_user_display_and_exists($acc2);

$has_any_existing = ($m_exists || $a1_exists || $a2_exists);

// Règle d’affichage des bruts :
// - admin/code-maitre => toujours
// - OU s'il y a au moins un conseiller existant sur la ligne
$can_show_raw_names = $can_edit || $has_any_existing;

$conseillers_list = [];

// mandataire
if ($m_name) $conseillers_list[] = $m_name;
elseif ($can_show_raw_names && $mandataire_brut) $conseillers_list[] = $mandataire_brut;

// acc1
if ($a1_name) $conseillers_list[] = $a1_name;
elseif ($can_show_raw_names && $acc1_brut) $conseillers_list[] = $acc1_brut;

// acc2
if ($a2_name) $conseillers_list[] = $a2_name;
elseif ($can_show_raw_names && $acc2_brut) $conseillers_list[] = $acc2_brut;

$conseillers = implode(', ', array_filter($conseillers_list));


// Frais min/max + taux_commission (robuste)
$frais_min = $frais_max = 0;
$taux_commission_percent = null; // en % (ex : 30 pour 30%)
$page_taux = 77;
$taux_de_commissions = get_field('taux_de_commission', $page_taux) ?: [];

if ($taux_de_commissions && $contrat_id) {
    foreach ($taux_de_commissions as $t) {
        $nom_contrat_rep = isset($t['nom_du_contrat']) ? (string)$t['nom_du_contrat'] : '';

        // Normalisations pour comparaison
        $nom_norm = trim($nom_contrat_rep);
        $contrat_norm = is_scalar($contrat_id) ? trim((string)$contrat_id) : '';
        $contrat_title = is_numeric($contrat_id) ? trim(get_the_title(intval($contrat_id))) : '';

        // Match si le nom du répéteur correspond soit au champ stocké, soit au title du post
        if ($nom_norm !== '' && ($nom_norm === $contrat_norm || ($contrat_title !== '' && $nom_norm === $contrat_title))) {
            $frais_min = floatval($t['frais_min'] ?? 0);
            $frais_max = floatval($t['frais_max'] ?? 0);
            $taux_commission_percent = floatval($t['taux_commission'] ?? 0);
            break;
        }
    }
}

// Calcul du ratio (comme avant)
$ratio = 0;
if ($frais_max > $frais_min && $frais_user >= $frais_min) {
    $ratio = min(1, ($frais_user - $frais_min) / ($frais_max - $frais_min));
}

// Si on n'a pas trouvé de taux_commission, on log pour debug et on retombe sur 20% (comportement antérieur)
// => change la valeur par défaut si tu veux (ex: 0)
if ($taux_commission_percent === null || $taux_commission_percent == 0) {
    error_log("DEBUG COMM: Pas de taux_commission trouvé pour contrat '{$contrat_id}' (post?) sur page $page_taux — fallback 20%");
    $taux_commission_percent = 20; // fallback pour compatibilité avec l'ancien code (0.2)
}

// Commission finale = volume * (taux_commission% / 100) * ratio
$commission = round($volume * ($taux_commission_percent / 100) * $ratio, 2);


    $dt = DateTime::createFromFormat('Y-m-d', $date) ?: DateTime::createFromFormat('d/m/Y', $date);
    $year = $dt ? $dt->format('Y') : '';
  ?>
    <tr data-year="<?= esc_attr($year) ?>">
     <td>
<?php if ($can_edit): ?>
    <select name="ventes[<?= $id ?>][statut_vente]">
        <?php foreach (['EN COURS','ARBITRAGE','INSTANCE','PAYÉ','RÉSILIÉ','TRANSFERT EN COURS'] as $opt): ?>
            <option value="<?= esc_attr($opt) ?>" <?= selected($opt,$statut,false) ?>>
                <?= esc_html($opt) ?>
            </option>
        <?php endforeach; ?>
    </select>
<?php else: ?>
    <strong><?= esc_html($statut) ?></strong>
<?php endif; ?>
</td>

      <td><?= esc_html($date) ?></td>
      <td><?= esc_html($client) ?></td>
      <td><?= esc_html($contrat_id) ?></td>
      <td><?= esc_html($conseillers) ?></td>
      <td class="editable">
<?php if ($can_edit): ?>
    <input type="number" step="0.01" 
           name="ventes[<?= $id ?>][volume_total_signe]" 
           value="<?= esc_attr($volume) ?>">
<?php else: ?>
    <?= esc_html($volume) ?>
<?php endif; ?>
</td>
      <td><?= esc_html($frais_user) ?>%</td>
      <td><?= esc_html(number_format($commission, 2, ',', ' ')) ?> €</td>
      <td>
        <div class="tooltip-container">
          <span class="tooltip-icon">ℹ️</span>
          <div class="tooltip-box">
            <strong>UCS :</strong> <?= ($sagit_ucs === 'Oui' ? 'Oui' : 'Non') ?><br>
            <em>Plus d’infos à venir...</em>
          </div>
        </div>
      </td>
      <td>
<?php if ($can_edit): ?>
    <span class="delete-icon" data-id="<?= $id ?>">🗑️</span>
<?php endif; ?>
</td>
    </tr>
  <?php endforeach; ?>
  </tbody>
</table>
<div class="pagination-wrapper" id="ventesPagination"></div>
<?php if ($can_edit): ?>
<p style="text-align:right; margin:10px 20px;">
  <button type="submit" name="save_ventes" class="button-validate">ENREGISTRER</button>
</p>
<?php endif; ?>
<div class="pagination-wrapper" id="ventesPagination"></div>
</form>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const table = document.querySelector('.liste-ventes-table');
    if (!table) return;

    const tbody = table.querySelector('tbody');
    const perPage = 50;
    let currentPage = 1;

    // Récupération des lignes et ajout data-year si nécessaire
    const rows = Array.from(tbody.querySelectorAll('tr')).map(tr => {
        const year = tr.dataset.year || '';
        tr.dataset.year = year;
        return tr;
    });

    // Filtre année
    const selectAnnee = document.getElementById('annee-select');
    if (selectAnnee) {
        selectAnnee.addEventListener('change', function() {
            currentPage = 1;
            renderTable();
        });
    }

    // Poubelle
    document.querySelectorAll('.delete-icon').forEach(btn => {
        btn.addEventListener('click', () => {
            const id = btn.dataset.id;
            if (!confirm("Supprimer cette vente ?")) return;
            fetch("<?= admin_url('admin-ajax.php') ?>", {
                method: 'POST',
                body: new URLSearchParams({action:'delete_vente', id})
            }).then(() => location.reload());
        });
    });

    // Pagination
    const paginationWrapper = document.getElementById('ventesPagination');

    function renderTable() {
        tbody.innerHTML = '';
        const selectedYear = selectAnnee ? selectAnnee.value : '';
        const filtered = rows.filter(r => !selectedYear || r.dataset.year === selectedYear);
        const totalPages = Math.ceil(filtered.length / perPage);

        const start = (currentPage - 1) * perPage;
        const pageRows = filtered.slice(start, start + perPage);

        pageRows.forEach(r => tbody.appendChild(r));

        // Pagination boutons
        if (paginationWrapper) {
            paginationWrapper.innerHTML = '';
            const createBtn = (text, page, disabled = false, active = false) => {
                const sp = document.createElement('span');
                sp.textContent = text;
                if (disabled) sp.classList.add('disabled');
                if (active) sp.classList.add('active');
                if (!disabled && !active) sp.addEventListener('click', () => {
                    currentPage = page;
                    renderTable();
                });
                return sp;
            };

            const visiblePages = 3;
            let startPage = Math.max(1, currentPage - Math.floor(visiblePages/2));
            let endPage = Math.min(totalPages, startPage + visiblePages - 1);
            if(endPage - startPage < visiblePages - 1) startPage = Math.max(1, endPage - visiblePages + 1);

            paginationWrapper.appendChild(createBtn('DÉBUT', 1, currentPage===1));
            paginationWrapper.appendChild(createBtn('«', currentPage-1, currentPage===1));
            for (let i=startPage; i<=endPage; i++) paginationWrapper.appendChild(createBtn(i, i, false, i===currentPage));
            paginationWrapper.appendChild(createBtn('»', currentPage+1, currentPage===totalPages));
            paginationWrapper.appendChild(createBtn('FIN', totalPages, currentPage===totalPages));
        }
    }

    renderTable();
});
</script>



<?php
    return ob_get_clean();
}

add_action('wp_ajax_delete_vente', function(){
  $id = intval($_POST['id']);
  if ($id && current_user_can('delete_post', $id)) {
      wp_delete_post($id, true);
      wp_send_json_success();
  }
  wp_send_json_error();
});

add_action('save_post', function($post_id, $post, $update) {
    if (wp_is_post_revision($post_id) || $post->post_type !== 'post') return;
    $form_source = get_post_meta($post_id, 'admin_form_source', true);
    if ($form_source !== 'elementorb9dc4a7') return;
    if (!wp_next_scheduled('add_historique_vente', [$post_id])) {
        wp_schedule_single_event(time() + 2, 'add_historique_vente', [$post_id]);
    }
}, 10, 3);

add_action('add_historique_vente', function($post_id) {

    $post = get_post($post_id);
    if (!$post) return;

    // Récupération des champs ACF
    $mandataire = get_field('mandataire', $post_id) ?: '';
    $acc1_id = get_field('accompagnant', $post_id);
    $acc2_id = get_field('second_accompagnant', $post_id);
    $contrat = get_field('veuillez_selectionner_le_contrat_souscrit', $post_id) ?: '';
    $volume = get_field('volume_total_signe', $post_id) ?: 0;

    // Fonction utilitaire pour récupérer le display_name si ID
    $get_user_name = function($id_or_text) {
        if (empty($id_or_text) || $id_or_text === '-') return '';
        if (is_numeric($id_or_text)) {
            $user = get_userdata(intval($id_or_text));
            if ($user) return $user->display_name;
        }
        return $id_or_text;
    };

    // Récupérer tous les conseillers
    $conseillers_array = [
        $get_user_name($mandataire),
        $get_user_name($acc1_id),
        $get_user_name($acc2_id),
    ];

    // Supprimer les vides
    $conseillers_array = array_filter($conseillers_array);
    $conseillers = implode(', ', $conseillers_array);

    // Formater le volume avec espace pour milliers et 2 décimales
    $volume_formate = number_format(floatval($volume), 2, ',', ' ');

    // Préparer le texte historique
    $historique_text = '<span style="margin-right:12px;">📂</span>'
                      . '<span style="margin-right:4px;">Nouvelle vente:</span>'
                      . '<span style="margin-right:8px;">' . esc_html($contrat) . '</span>'
                      . esc_html($volume_formate) . ' €'
                      . '<span style="font-size:0.85em; margin-left:5px;">(' . esc_html($conseillers) . ')</span>';

    // Récupérer l'historique existant
    $historique = get_field('historique_rep', 'option');
    if (!is_array($historique)) $historique = [];

    // Ajouter la nouvelle ligne
    $historique[] = [
        'historique_sous' => $historique_text,
        'timestamp' => current_time('Y-m-d H:i:s'),
    ];

    // Sauvegarder
    update_field('historique_rep', $historique, 'option');

    error_log("DEBUG HISTO: Ligne historique ajoutée pour post_id=$post_id avec conseillers: " . print_r($conseillers_array,true));

}, 10, 1);