<?php
if (!defined('ABSPATH')) exit;

/**
 * Trouve un attachment ID en recherchant le NOM DE FICHIER dans la médiathèque.
 * Ex: "abeille.jpg" -> 1234
 */
function arc_find_attachment_id_by_filename(string $filename): int {
    global $wpdb;

    $filename = trim($filename);
    if ($filename === '') return 0;

    // Essai via meta _wp_attached_file (le plus fiable)
    $like = '%' . $wpdb->esc_like($filename);
    $id = (int) $wpdb->get_var($wpdb->prepare(
        "SELECT post_id
         FROM {$wpdb->postmeta}
         WHERE meta_key = '_wp_attached_file'
         AND meta_value LIKE %s
         ORDER BY post_id DESC
         LIMIT 1",
        $like
    ));
    if ($id) return $id;

    // Fallback via GUID (moins fiable mais utile)
    $like2 = '%' . $wpdb->esc_like('/' . $filename);
    $id = (int) $wpdb->get_var($wpdb->prepare(
        "SELECT ID
         FROM {$wpdb->posts}
         WHERE post_type='attachment'
         AND guid LIKE %s
         ORDER BY ID DESC
         LIMIT 1",
        $like2
    ));

    return $id ?: 0;
}

/**
 * Import (sideload) un fichier du plugin vers la médiathèque, retourne attachment_id ou 0.
 */
function arc_import_media_from_plugin(string $file_path, string $filename): int {

    // Nécessaire pour media_handle_sideload
    require_once ABSPATH . 'wp-admin/includes/file.php';
    require_once ABSPATH . 'wp-admin/includes/media.php';
    require_once ABSPATH . 'wp-admin/includes/image.php';

    if (!file_exists($file_path)) {
        return 0;
    }

    $tmp = wp_tempnam($filename);
    if (!$tmp) return 0;

    if (!copy($file_path, $tmp)) {
        @unlink($tmp);
        return 0;
    }

    $file = [
        'name'     => $filename,
        'tmp_name' => $tmp,
    ];

    $attachment_id = media_handle_sideload($file, 0);

    if (is_wp_error($attachment_id)) {
        @unlink($tmp);
        return 0;
    }

    return (int) $attachment_id;
}

/**
 * Fonction générique : mappe depuis médiathèque, sinon importe depuis assets/images/
 * - $items: [ 'slug' => 'filename.png', ... ]
 * - $option_prefix: ex 'arc_logo_id_' ou 'arc_media_' (sans le suffixe _id)
 * - $title_prefix: ex 'LOGO' ou 'ARC MEDIA'
 */
function arc_map_or_import_media(array $items, string $option_prefix, string $title_prefix = ''): void {

    foreach ($items as $key => $filename) {

        $key = (string) $key;
        $filename = (string) $filename;

        // Définition de la clé option
        // logos: arc_logo_id_{slug}
        // media: arc_media_{key}_id
        if ($option_prefix === 'arc_logo_id_') {
            $option_key = 'arc_logo_id_' . $key;
        } else {
            $option_key = 'arc_media_' . $key . '_id';
        }

        // 1) Si déjà OK, on skip
        $existing_id = (int) get_option($option_key, 0);
        if ($existing_id && get_post($existing_id)) {
            continue;
        }

        // 2) Tentative de mapping dans la médiathèque
        $found_id = arc_find_attachment_id_by_filename($filename);
        if ($found_id && get_post($found_id)) {
            update_option($option_key, $found_id);

            // Optionnel: titre lisible
            if ($title_prefix !== '') {
                wp_update_post([
                    'ID'         => $found_id,
                    'post_title' => $title_prefix . ' - ' . strtoupper($key),
                ]);
            }
            continue;
        }

        // 3) Sinon, on importe depuis le plugin (assets/images/)
        $file_path = ARC_CORE_PATH . 'assets/images/' . $filename;
        if (!file_exists($file_path)) {
            error_log('[arc MEDIA] introuvable (plugin): ' . $file_path);
            continue;
        }

        $attachment_id = arc_import_media_from_plugin($file_path, $filename);
        if (!$attachment_id) {
            error_log('[arc MEDIA] import échoué: ' . $filename);
            continue;
        }

        update_option($option_key, $attachment_id);

        if ($title_prefix !== '') {
            wp_update_post([
                'ID'         => $attachment_id,
                'post_title' => $title_prefix . ' - ' . strtoupper($key),
            ]);
        }
    }
}

/**
 * Fonction principale : logos + media non-logos
 * -> peut être appelée par ton installer OU par un outil "sync"
 */
function arc_install_logos_media() {

    // slug => filename (LOGOS)
    $logos = [
        'abeille'        => 'abeille.jpg',
        'add-value'      => 'add-value.png',
        'adf'            => 'adf.png',
        'afi-esca'       => 'afi-esca.png',
        'airbag'         => 'airbag.jpg',
        'allianz'        => 'allianz.jpg',
        'alptis'         => 'alptis.jpg',
        'ami3f'          => 'ami3f.jpg',
        'apicil'         => 'apicil.jpg',
        'apivia'         => 'apivia.jpg',
        'april'          => 'april.png',
        'assurea'        => 'assurea.jpg',
        'assurmax'       => 'assurmax.jpg',
        'axa-thema'      => 'axa-thema.png',
        'axa'            => 'axa.jpg',
        'cardif'         => 'cardif.jpg',
        'dattak'         => 'dattak.jpg',
        'eca'            => 'eca.png',
        'entoria'        => 'entoria.png',
        'equitim'        => 'equitim.png',
        'fma'            => 'fma.jpg',
        'garance'        => 'garance.png',
        'generali'       => 'generali.jpg',
        'groupe-leader'  => 'groupe-leader.jpg',
        'harmonie'       => 'harmonie.jpg',
        'helvetia'       => 'helvetia.jpg',
        'henner'         => 'henner.jpg',
        'hodeva'         => 'hodeva.png',
        'intencial'      => 'intencial.png',
        'kereis'         => 'kereis.jpg',
        'malakoff'       => 'malakoff.jpg',
        'maxance'        => 'maxance.jpg',
        'metlife'        => 'metlife.png',
        'mma'            => 'mma.jpg',
        'neoliane'       => 'neoliane.jpg',
        'netvox'         => 'netvox.jpg',
        'novelia'        => 'novelia.png',
        'oddo'           => 'oddo.jpg',
        'onelife'        => 'onelife.png',
        'optimum'        => 'optimum.png',
        'selencia'       => 'selencia.jpg',
        'sollyazar'      => 'sollyazar.jpg',
        'sp-vie'         => 'sp-vie.png',
        'spring'         => 'spring.jpg',
        'stoik'          => 'stoik.jpg',
        'swisslife'      => 'swisslife.jpg',
        'tetris'         => 'tetris.jpg',
        'zephir'         => 'zephir.png',
    ];

    // key => filename (MEDIA NON-LOGOS)
    $media = [
        'logo_white'        => 'ARC-PARTNERS-SOLUTIONS-LOGO-WHITE.png',
        'logo_arc'          => 'ARC-PARTNERS-SOLUTIONS-LOGO.png',
        'lettre_ass_vie'    => 'lettre-de-mission-assurance-vie.png',
        'lettre_prev'       => 'lettre-de-mission-prevoyance.png',
        'lettre_retraite'   => 'lettre-de-mission-retraite.png',
        'lettre_mut'        => 'lettre-de-mission-sante.png',
        'rapport_ass_vie'   => 'rapport-de-mission-assurance-vie.png',
        'rapport_prev'      => 'rapport-de-mission-prevoyance.png',
        'rapport_retraite'  => 'rapport-de-mission-retraite.png',
        'rapport_mut'       => 'rapport-de-mission-sante.png',
        'suivi_ass_vie'     => 'suivi-de-mission-assurance-vie.png',
        'suivi_prev'        => 'suivi-de-mission-prevoyance.png',
        'suivi_retraite'    => 'suivi-de-mission-retraite.png',
        'suivi_mut'         => 'suivi-de-mission-sante.png',
        'background_login'  => 'pexels-photo-3756679.jpg',
        'tab_gris'          => 'Z5XOwv4.png',
        'tab_niveau'        => 'suivi-assurance-vie-3-e1727771834828.png',
        // ajoute ici orientation etc si besoin
        // 'orientation'     => 'orientation.png',
    ];

    // 1) Logos
    arc_map_or_import_media($logos, 'arc_logo_id_', 'LOGO');

    // 2) Medias
    arc_map_or_import_media($media, 'arc_media_', 'ARC MEDIA');
}
