<?php
/*
Plugin Name: Conformité PDF Generator (FEA + AJAX + DocuSign FINAL)
Description: Génère un PDF Conformité après FEA, avec option Télécharger ou Envoyer par DocuSign.
Version: 7.0
*/
if (defined('CPG_PLUGIN_LOCK')) {
    error_log("[CPG] 🔥 Plugin déjà chargé — instance ignorée");
    return;
}
define('CPG_PLUGIN_LOCK', true);

error_log("[CPG] Plugin instance initialisée");




if (!defined('ABSPATH')) exit;

/*--------------------------------------------------
  0) CONFIG DOCUSIGN
---------------------------------------------------*/

// Clé d’intégration
define('CPG_DK_INTEGRATION_KEY',     '789b83b1-3066-4dc6-a4a8-b835cb3870d3');

// User ID GUID (UTILISATEUR, pas l’app)
define('CPG_DK_USER_ID',             'ffb4cacd-bfb7-43c3-bee2-e25186f6beb4');

// Account ID
define('CPG_DK_ACCOUNT_ID',          '990512ac-4b4a-4318-a456-658679e2da89');

// API URLs (PROD EU)
define('CPG_DK_BASE_PATH',   'https://eu.docusign.net/restapi');
define('CPG_DK_AUTH_SERVER', 'https://account.docusign.com');

// Clé privée RSA
define('CPG_DK_PRIVATE_KEY_PATH', WP_CONTENT_DIR . '/uploads/docusign_private_key.pem');


/*--------------------------------------------------
  1) SESSION
---------------------------------------------------*/
add_action('init', function() {
    if (!session_id()) {
        @session_start();
    }
});


/*--------------------------------------------------
  2) CHARGER MPDF
---------------------------------------------------*/
$autoload = plugin_dir_path(__FILE__) . 'vendor/autoload.php';
if (file_exists($autoload)) {
    require_once $autoload;
    error_log("[CPG][BOOT] mPDF chargé.");
} else {
    error_log("[CPG][BOOT] ❌ mPDF introuvable !");
}


/*--------------------------------------------------
  3) GÉNÉRATION PDF
---------------------------------------------------*/
function cpg_debug_generate_pdf($post_id) {

    $post_id = intval($post_id);
    error_log("[CPG][PDF] Début génération PDF pour post_id=$post_id");

    $post = get_post($post_id);
    if (!$post) {
        error_log("[CPG][PDF] ❌ post introuvable.");
        return false;
    }

    /**
     * 1) Charger le template et récupérer $html + $footer_html
     */
// --- Sélection du template selon type_confo ---
$type_confo = get_field('type_confo', $post_id);
$type_confo = is_array($type_confo) ? reset($type_confo) : trim((string)$type_confo);

error_log("[CPG][PDF] type_confo = " . var_export($type_confo, true));

$template_file = '';

if ($type_confo === 'DER IARD') {
    $template_file = 'pdf-template-iard.php';
} elseif ($type_confo === 'LETTRES VIE / RETRAITE') {
    $template_file = 'pdf-template.php';
} elseif ($type_confo === 'DER VIE / RETRAITE') {
    $template_file = 'pdf-template-der-vie.php';
} else {
    // Cas "&nbsp;" ou vide → on ne fait rien
    error_log("[CPG][PDF] ⛔ type_confo vide ou invalide → PDF non généré");
    return false;
}

// --- Chargement du template ---
ob_start();
include plugin_dir_path(__FILE__) . 'templates/' . $template_file;
$html = ob_get_clean();


    if (!trim($html)) {
        error_log("[CPG][PDF] ❌ HTML vide !");
        return false;
    }

    /**
     * 2) Créer instance mPDF
     *    Ajout de marges adaptées au footer
     */
    try {

        $mpdf = new \Mpdf\Mpdf([
            'margin_top'    => 20,
            'margin_right'  => 15,
            'margin_bottom' => 25,  // important pour que le footer respire
            'margin_left'   => 15,
        ]);

        // Si footer fourni par le template → l’injecter ici
        if (isset($footer_html) && trim($footer_html)) {
            $mpdf->SetHTMLFooter($footer_html);
            error_log("[CPG][PDF] ✔ Footer appliqué au PDF.");
        } else {
            error_log("[CPG][PDF] ⚠ Aucun footer trouvé dans le template.");
        }

        $private = plugin_dir_path(__FILE__) . 'private/';
        if (!file_exists($private)) {
            mkdir($private, 0755, true);
        }

        $pdf_path = $private . "conformite-$post_id.pdf";

        /**
         * 3) Génération du PDF
         */
        $mpdf->WriteHTML($html);

        // Sauvegarde (F = File)
        $mpdf->Output($pdf_path, 'F');

        error_log("[CPG][PDF] ✔ PDF généré : $pdf_path");

        return $pdf_path;

    } catch (Exception $e) {

        error_log("[CPG][PDF] ❌ EXCEPTION PDF : " . $e->getMessage());
        return false;
    }
}



/*--------------------------------------------------
  4) AJAX : GÉNÉRER PDF (+ éventuel DocuSign)
---------------------------------------------------*/
add_action('wp_ajax_cpg_generate_pdf', function() {

    $post_id = intval($_POST['post_id'] ?? 0);
    error_log("[CPG][AJAX:PDF] Reçu AJAX PDF pour post_id=$post_id");

    $resp = [
        'success'   => false,
        'url'       => '',
        'docusign'  => false,
        'envelope_id' => null,
    ];

    if (!$post_id) {
        wp_send_json($resp);
    }

    $pdf = cpg_debug_generate_pdf($post_id);

    if (!$pdf) {
        wp_send_json($resp);
    }

    // URL publique du PDF
    $resp['success'] = true;
    $resp['url'] = cpg_build_secure_pdf_url($post_id, 300); // 5 min


    // Vérifier action_pdf côté serveur
    $action_pdf = get_field('action_pdf', $post_id);
    if (is_array($action_pdf)) {
        $action_pdf = reset($action_pdf);
    }
    $act_norm = mb_strtolower(trim((string)$action_pdf), 'UTF-8');
    error_log("[CPG][AJAX:PDF] action_pdf (serveur) = ".$act_norm);

    // Si on a choisi Docusign, on envoie l’enveloppe ici
    if (strpos($act_norm, 'docusign') !== false) {
        $env_id = cpg_dk_send_envelope($post_id);
        $resp['docusign']    = (bool)$env_id;
        $resp['envelope_id'] = $env_id ?: null;
    }

    wp_send_json($resp);
});


/*--------------------------------------------------
  5) AJAX : RÉCUPÉRER DERNIER POST
---------------------------------------------------*/
add_action('wp_ajax_cpg_last_conformite', function() {

    $uid = get_current_user_id();
    error_log("[CPG][AJAX:LAST] Recherche dernier post utilisateur $uid");

    $p = get_posts([
        'post_type'      => 'conformite',
        'posts_per_page' => 1,
        //'author'         => $uid,
        'orderby'        => 'modified',
        'order'          => 'DESC'
    ]);

    if (!$p) {
        error_log("[CPG][AJAX:LAST] ❌ aucun post");
        wp_send_json(['success' => false]);
    }

    $post_id    = $p[0]->ID;
    $action_pdf = get_field('action_pdf', $post_id);

    error_log("[CPG][AJAX:LAST] ✔ post_id=$post_id | action_pdf=".print_r($action_pdf,true));

    wp_send_json([
        'success'    => true,
        'post_id'    => $post_id,
        'action_pdf' => is_array($action_pdf)? reset($action_pdf): $action_pdf
    ]);
});

/*--------------------------------------------------
  6bis) URL SIGNÉE (secure) pour le PDF
---------------------------------------------------*/
function cpg_build_secure_pdf_url($post_id, $ttl_seconds = 300) {

    $post_id = intval($post_id);
    $user_id = get_current_user_id();
    if (!$user_id || !$post_id) return '';

    $exp = time() + intval($ttl_seconds);

    // Donnée signée : post + user + exp
    $payload = $post_id . '|' . $user_id . '|' . $exp;

    // Signature HMAC avec un secret WP (AUTH_SALT ou AUTH_KEY)
    $sig = hash_hmac('sha256', $payload, wp_salt('auth'));

    // URL "longue et random" (signature)
    // Format: /download-conformite/<post_id>/<exp>/<sig>/
    return home_url("/download-conformite/{$post_id}/{$exp}/{$sig}/");
}


/*--------------------------------------------------
  6) ENDPOINT PDF (URL signée)
---------------------------------------------------*/
register_activation_hook(__FILE__, function () {
    cpg_add_rewrite_rules();
    flush_rewrite_rules();
});

register_deactivation_hook(__FILE__, function () {
    flush_rewrite_rules();
});

function cpg_add_rewrite_rules() {
    // /download-conformite/<post_id>/<exp>/<sig>/
    add_rewrite_rule(
        '^download-conformite/([0-9]+)/([0-9]+)/([a-f0-9]{64})/?$',
        'index.php?download_conformite=1&post_id=$matches[1]&cpg_exp=$matches[2]&cpg_sig=$matches[3]',
        'top'
    );
}
add_action('init', 'cpg_add_rewrite_rules');

add_filter('query_vars', function($vars){
    $vars[] = 'download_conformite';
    $vars[] = 'post_id';
    $vars[] = 'cpg_exp';
    $vars[] = 'cpg_sig';
    return $vars;
});

add_action('template_redirect', function(){

    if (intval(get_query_var('download_conformite')) !== 1) return;

    // 1) Auth obligatoire (sinon n’importe qui avec le lien peut tenter)
    if (!is_user_logged_in()) {
        status_header(403);
        wp_die("Accès refusé.");
    }

    $post_id = intval(get_query_var('post_id'));
    $exp     = intval(get_query_var('cpg_exp'));
    $sig     = sanitize_text_field(get_query_var('cpg_sig'));

    if (!$post_id || !$exp || !$sig) {
        status_header(400);
        wp_die("Lien invalide.");
    }

    // 2) Expiration
    if (time() > $exp) {
        status_header(410);
        wp_die("Lien expiré. Merci de régénérer le PDF.");
    }

    // 3) Vérif signature (liée à l'utilisateur courant)
    $user_id = get_current_user_id();
    $payload = $post_id . '|' . $user_id . '|' . $exp;
    $expected = hash_hmac('sha256', $payload, wp_salt('auth'));

    if (!hash_equals($expected, $sig)) {
        status_header(403);
        wp_die("Signature invalide.");
    }

    // 4) Optionnel mais recommandé : vérifier que l’utilisateur a le droit sur ce post
    $post = get_post($post_id);
    if (!$post) {
        status_header(404);
        wp_die("Document introuvable.");
    }
    // Si tu veux restreindre aux auteurs :
    if (intval($post->post_author) !== $user_id && !current_user_can('edit_post', $post_id)) {
        status_header(403);
        wp_die("Non autorisé.");
    }

    // 5) Servir le fichier
    $file = plugin_dir_path(__FILE__)."private/conformite-$post_id.pdf";
    if (!file_exists($file)) {
        status_header(404);
        wp_die("PDF introuvable.");
    }

    header("Content-Type: application/pdf");
    header("Content-Disposition: inline; filename=conformite-$post_id.pdf");
    header("Content-Length: " . filesize($file));
    readfile($file);
    exit;
});


/*--------------------------------------------------
  7) DOCUSIGN : JWT
---------------------------------------------------*/
function cpg_dk_jwt_token() {

    error_log("[CPG][DK][JWT] Début génération token…");

    if (!file_exists(CPG_DK_PRIVATE_KEY_PATH)) {
        error_log("[CPG][DK][JWT] ❌ Fichier clé privée introuvable : " . CPG_DK_PRIVATE_KEY_PATH);
        return false;
    }

    $private = file_get_contents(CPG_DK_PRIVATE_KEY_PATH);
    if (!trim($private)) {
        error_log("[CPG][DK][JWT] ❌ Clé privée vide.");
        return false;
    }

    error_log("[CPG][DK][JWT] Longueur clé privée = ".strlen($private)." octets");

    $now = time();

    $header = ['alg' => 'RS256', 'typ' => 'JWT'];
    $claims = [
        'iss'   => CPG_DK_INTEGRATION_KEY,
        'sub'   => CPG_DK_USER_ID,
        'aud'   => 'account.docusign.com',
        'iat'   => $now,
        'exp'   => $now + 3600,
        'scope' => 'signature impersonation'
    ];

    error_log("[CPG][DK][JWT] Claims utilisés : ".print_r($claims, true));

    $seg = function($d){
        return rtrim(strtr(base64_encode(json_encode($d)), '+/','-_'),'=');
    };

    $unsigned = $seg($header).".".$seg($claims);

    $signature = '';
    $ok = openssl_sign($unsigned, $signature, $private, 'sha256');

    error_log("[CPG][DK][JWT] openssl_sign = ".($ok ? 'OK' : 'FAIL'));

    if (!$ok) {
        if (function_exists('openssl_error_string')) {
            error_log("[CPG][DK][JWT] OpenSSL error: ".openssl_error_string());
        }
        return false;
    }

    $jwt = $unsigned.".".rtrim(strtr(base64_encode($signature),'+/','-_'),'=');

    $resp = wp_remote_post(CPG_DK_AUTH_SERVER."/oauth/token",[
        'headers' => ['Content-Type'=>'application/x-www-form-urlencoded'],
        'body'    => [
            'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
            'assertion'  => $jwt
        ]
    ]);

    error_log("[CPG][DK][JWT] RAW HTTP : ".print_r($resp,true));

    if (is_wp_error($resp)) {
        error_log("[CPG][DK][JWT] ❌ WP_Error : ".$resp->get_error_message());
        return false;
    }

    $body = json_decode(wp_remote_retrieve_body($resp),true);

    error_log("[CPG][DK][JWT] BODY : ".print_r($body,true));

    return $body['access_token'] ?? false;
}


/*--------------------------------------------------
  8) DOCUSIGN : ENVOI ENVELOPPE
---------------------------------------------------*/
function cpg_dk_send_envelope($post_id){

    $post_id = intval($post_id);
    error_log("[CPG][DK] Début envoi enveloppe pour post_id=$post_id");

    $pdf_path = plugin_dir_path(__FILE__)."private/conformite-$post_id.pdf";
    if (!file_exists($pdf_path)) {
        error_log("[CPG][DK] ❌ PDF introuvable.");
        return false;
    }

    $token = cpg_dk_jwt_token();
    if (!$token) {
        error_log("[CPG][DK] ❌ Token JWT impossible.");
        return false;
    }

    error_log("[CPG][DK] ✔ Token obtenu.");

    $client_email = get_field('mail',          $post_id);
    $cons_email   = get_field('conseiller_vie',$post_id);

    error_log("[CPG][DK] Emails : client=$client_email | conseiller=$cons_email");

    $pdf_b64 = base64_encode(file_get_contents($pdf_path));

    $env = [
        'emailSubject' => "Conformité à signer",
        'documents' => [[
            'documentBase64' => $pdf_b64,
            'name'           => "Conformité #$post_id",
            'fileExtension'  => 'pdf',
            'documentId'     => '1'
        ]],
        'recipients' => [
            'signers' => [
                [
                    'email'        => $client_email,
                    'name'         => $client_email,
                    'recipientId'  => '1',
                    'routingOrder' => '1',
                    'tabs'         => [
                        'signHereTabs' => [
                            ['anchorString' => '/signature_client/']
                        ]
                    ]
                ],
                [
                    'email'        => $cons_email,
                    'name'         => $cons_email,
                    'recipientId'  => '2',
                    'routingOrder' => '2',
                    'tabs'         => [
                        'signHereTabs' => [
                            ['anchorString' => '/signature_conseiller/']
                        ]
                    ]
                ]
            ]
        ],
        'status' => 'sent'
    ];

    $url = CPG_DK_BASE_PATH."/v2.1/accounts/".CPG_DK_ACCOUNT_ID."/envelopes";

    error_log("[CPG][DK] ENVELOPPE JSON : ".json_encode($env));

    $resp = wp_remote_post($url,[
        'headers' => [
            'Authorization' => 'Bearer '.$token,
            'Content-Type'  => 'application/json'
        ],
        'body' => json_encode($env)
    ]);

    error_log("[CPG][DK] RAW RESPONSE : ".print_r($resp,true));

    if (is_wp_error($resp)) return false;

    $body = json_decode(wp_remote_retrieve_body($resp),true);

    error_log("[CPG][DK] BODY : ".print_r($body,true));

    return $body['envelopeId'] ?? false;
}


/*--------------------------------------------------
  9) JS — AJAX STOP + LOADER + ANTI-REDIRECT
---------------------------------------------------*/

add_action('wp_footer', function () {

    // Charger seulement sur la page FEA Conformité
    if (!is_page(3187)) return;
?>
<script>
console.log("🔎 SYSTEME PDF AJAX + LOADER + DOCUSIGN chargé");

(function($){

    function CPG_LOG(){ console.log.apply(console, ["🔵[CPG JS]"].concat([...arguments])); }

    let CPG_LOCK = false;

    /* ======================================================
       1) LOADER (UTILISABLE AVANT & APRÈS REFRESH)
    ====================================================== */
    const style = document.createElement("style");
    style.innerHTML = `
#cpg-pdf-loader{
    position:fixed;
    inset:0;
    background:rgba(255,255,255,.85);
    display:flex;
    flex-direction:column;
    align-items:center;
    justify-content:center;
    z-index:99999;
    font-family:Arial,sans-serif;
}
#cpg-spinner{
    width:55px;
    height:55px;
    border:5px solid #3498db;
    border-top-color:transparent;
    border-radius:50%;
    animation:cpg-spin .9s linear infinite;
}
@keyframes cpg-spin{to{transform:rotate(360deg)}}
#cpg-loader-text{
    margin-top:18px;
    font-size:18px;
    color:#333;
    font-weight:500;
}`;
    document.head.appendChild(style);

    window.showLoader = function(text){
        if (document.getElementById("cpg-pdf-loader")) return;
        const d = document.createElement("div");
        d.id = "cpg-pdf-loader";
        d.innerHTML = `
            <div id="cpg-spinner"></div>
            <div id="cpg-loader-text">${text || "Génération du PDF…"}</div>
        `;
        document.body.appendChild(d);
    };

    window.hideLoader = function(){
        const d = document.getElementById("cpg-pdf-loader");
        if (d) d.remove();
    };

    /* ======================================================
       2) AVANT REFRESH — détecter fin FEA & stocker post_id
    ====================================================== */
    $(document).ajaxStop(function(){

        if (CPG_LOCK) return;

        // Stop si erreurs FEA / ACF
        if (document.querySelector('.acf-error, .fea-error, .acf-notice.-error')) {
            CPG_LOG("⛔ Erreurs détectées → arrêt");
            return;
        }

        CPG_LOCK = true;
        showLoader("Finalisation des données…");

        CPG_LOG("🚀 ajaxStop → récupération post_id");

        fetch("<?php echo admin_url('admin-ajax.php'); ?>?action=cpg_last_conformite")
            .then(r => r.json())
            .then(data => {

                CPG_LOG("last_conformite =", data);

                if (!data.success || !data.post_id) {
                    hideLoader();
                    CPG_LOG("❌ Aucun post_id trouvé");
                    return;
                }

                // 🔐 On stocke UNIQUEMENT ce qui est nécessaire
                sessionStorage.setItem("cpg_post_to_generate", data.post_id);
                sessionStorage.setItem("cpg_action_pdf", data.action_pdf || '');

                CPG_LOG("📦 post_id stocké pour après refresh :", data.post_id);
                hideLoader();
            });
    });

})(jQuery);


/* ======================================================
   3) APRÈS REFRESH — génération PDF + loader + popup
====================================================== */
(function(){

    const postId = sessionStorage.getItem("cpg_post_to_generate");
    if (!postId) return;

    sessionStorage.removeItem("cpg_post_to_generate");

    const actionPdf = (sessionStorage.getItem("cpg_action_pdf") || '').toLowerCase();
    sessionStorage.removeItem("cpg_action_pdf");

    console.log("🟢 POST-REFRESH : génération PDF post_id =", postId);

    // 👉 FEEDBACK IMMÉDIAT UTILISATEUR
    showLoader("Génération du PDF en cours…");

    const fd = new FormData();
    fd.append("action", "cpg_generate_pdf");
    fd.append("post_id", postId);

    fetch("<?php echo admin_url('admin-ajax.php'); ?>", {
        method: "POST",
        body: fd
    })
    .then(r => r.json())
    .then(pdf => {

        console.log("📄 PDF RESPONSE =", pdf);

        hideLoader();

        if (!pdf.success || !pdf.url) {
            alert("Erreur lors de la génération du PDF.");
            return;
        }

        // ----- MODE DOCUSIGN -----
        if (actionPdf.includes("docusign")) {
            alert(pdf.docusign
                ? "Enveloppe DocuSign envoyée."
                : "Erreur DocuSign (voir logs)");
            return;
        }

        // ----- MODE TÉLÉCHARGER -----
        showPopup(pdf.url);
    })
    .catch(() => {
        hideLoader();
        alert("Erreur technique lors de la génération du PDF.");
    });


    /* ======================================================
       4) POPUP PDF
    ====================================================== */
    function showPopup(pdfUrl){

        console.log("📄 Affichage popup PDF :", pdfUrl);

        const overlay = document.createElement("div");
        overlay.id = "cpg-modal-overlay";
        overlay.style = `
            position:fixed;
            inset:0;
            background:rgba(0,0,0,.55);
            display:flex;
            align-items:center;
            justify-content:center;
            z-index:999999;
        `;

        overlay.innerHTML = `
            <div style="
                background:#fff;
                padding:28px;
                border-radius:12px;
                width:420px;
                text-align:center;
                box-shadow:0 8px 25px rgba(0,0,0,.25);
                font-family:Arial,sans-serif;
            ">
                <h2 style="margin-top:0">Votre PDF est prêt 📄</h2>
                <p>Cliquez ci-dessous pour l’ouvrir.</p>
                <button id="cpg-open-pdf" style="
                    margin-top:18px;
                    background:#0073aa;
                    color:#fff;
                    padding:12px 20px;
                    border:none;
                    border-radius:8px;
                    cursor:pointer;
                    font-size:16px;
                ">
                    Ouvrir le PDF
                </button>
            </div>
        `;

        document.body.appendChild(overlay);

        document.getElementById("cpg-open-pdf").onclick = () => {
            const a = document.createElement("a");
            a.href = pdfUrl;
            a.target = "_blank";
            a.rel = "noopener noreferrer";
            document.body.appendChild(a);
            a.click();
            a.remove();
            overlay.remove();
        };
    }

})();
</script>



<?php
});















add_action('wp_enqueue_scripts', function () {
    // Charge le script uniquement sur le front
    wp_enqueue_script(
        'acf-client-selector',
        plugin_dir_url(__FILE__) . 'assets/js/acf-client-selector.js',
        ['jquery'],
        '1.0',
        true
    );

    wp_localize_script('acf-client-selector', 'cgAjax', [
        'url' => admin_url('admin-ajax.php'),
    ]);
});


add_action('wp_ajax_get_user_clients', 'cg_get_user_clients');
add_action('wp_ajax_nopriv_get_user_clients', 'cg_get_user_clients');

function cg_get_user_clients() {

    error_log("AJAX get_user_clients RUN");

    if (!is_user_logged_in()) {
        wp_send_json([]);
    }

    $user_id = get_current_user_id();

    // On récupère TOUS les clients et on filtre en PHP
    $posts = get_posts([
        'post_type'      => 'clients',
        'posts_per_page' => -1,
        'fields'         => 'ids'
    ]);

    $out = [];

    foreach ($posts as $p_id) {

        $saved_conseillers = get_post_meta($p_id, 'conseillers_manual', true);

        if (!empty($saved_conseillers) && is_array($saved_conseillers)) {

            if (in_array($user_id, $saved_conseillers)) {
                $out[] = [
                    'id'    => $p_id,
                    'title' => get_the_title($p_id)
                ];
            }
        }
    }

    wp_send_json($out);
}


// 3) AJAX #2 : récupérer les données ACF d’un client
add_action('wp_ajax_get_client_data', 'cg_get_client_data');
add_action('wp_ajax_nopriv_get_client_data', 'cg_get_client_data'); // safe

function cg_get_client_data() {

    if (!is_user_logged_in()) {
        wp_send_json_error(['message' => 'Not logged']);
    }

    $client_id = isset($_POST['client_id']) ? intval($_POST['client_id']) : 0;
    $user_id   = get_current_user_id();

    if (!$client_id) {
        wp_send_json_error(['message' => 'Missing client_id']);
    }

    // Vérification autorisation (client lié au conseiller via conseillers_manual)
    $owners = get_post_meta($client_id, 'conseillers_manual', true);

    if (empty($owners) || !is_array($owners) || !in_array($user_id, $owners)) {
        wp_send_json_error(['message' => 'Forbidden']);
    }

    // Correspondances ACF → ACF
    $mapping = [
        'civilite'               => 'titre',
        'nom_client'             => 'nom_client_ag',
        'prenom_client'          => 'prenom_client_ag',
        'date_naissance_client'  => 'date_naissance_client_ag',
        'lieu_naissance_client'  => 'lieu_de_naissance',
        'adresse'                => 'xrue_perso',
        'ville'                  => 'xville_perso',
        'cp'                     => 'xcode_postal_perso',
        'mail'                   => 'emailperso',
        'tel'                    => 'mobile',
        'profession_client'      => 'profession',
        'situation_fam_client'   => 'situation_familiale',
        'charge_client'          => 'nombre_denfant_a_charge',
        'statut_pat_client'      => 'le_client_est',
    ];

    $data = [];

    foreach ($mapping as $target => $source) {
        // get_field parce que ce sont des champs ACF
        $data[$target] = function_exists('get_field')
            ? get_field($source, $client_id)
            : get_post_meta($client_id, $source, true);
    }

    wp_send_json($data);
}
