<?php

// SPDX-FileCopyrightText: 2022-2025 Ovation S.r.l. <help@dynamic.ooo>
// SPDX-License-Identifier: LicenseRef-GPL-3.0-with-dynamicooo-additional-terms

namespace DynamicShortcodes\Core\Settings;

use DynamicShortcodes\Plugin;
use DynamicShortcodes\Core\LicenseFacade;

class Manager {
	const MAIN_PAGE_ID            = 'dynamic-shortcodes';
	const OPTIONS_PREFIX          = 'dynamic_shortcodes_';
	const OPTION_POWER_SHORTCODES = self::OPTIONS_PREFIX . '_power_shortcodes';

	/**
	 * @var AdminPages\Manager;
	 */
	public $admin_pages;

	public function __construct() {
		add_action( 'admin_menu', [ $this, 'add_menu_pages' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_scripts' ] );
		$this->admin_pages = new AdminPages\Manager();
		add_action( 'admin_init', [ $this, 'redirect_after_activation' ] );
	}

	public function redirect_after_activation() {
		if ( ! get_transient( 'dsh_do_activation_redirect' ) ) {
			return;
		}
		delete_transient( 'dsh_do_activation_redirect' );
		//phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( wp_doing_ajax() || is_network_admin() || isset( $_GET['activate-multi'] ) ) {
			return;
		}
		if ( get_option( 'dsh_onboarding_done', false ) ) {
			return;
		}
		wp_safe_redirect( admin_url( 'admin.php?page=' . Plugin::instance()->plugin_utils->get_license_admin_page() ) );
		exit;
	}

	public function enqueue_admin_scripts( $hook_suffix ) {
		wp_register_style(
			'dynamic-shortcodes-admin-notice',
			DYNAMIC_SHORTCODES_URL . 'assets/css/admin-notice.css',
			[],
			DYNAMIC_SHORTCODES_VERSION,
		);
		wp_enqueue_style( 'dynamic-shortcodes-admin-notice' );

		if ( strpos( $hook_suffix, self::MAIN_PAGE_ID ) !== 0 ) {
			return;
		}

		// Only on DSH pages
		wp_register_style( 'dsh-select2', plugins_url( 'assets/lib/select2/select2.min.css', DYNAMIC_SHORTCODES__FILE__ ), [], DYNAMIC_SHORTCODES_VERSION );
		wp_register_script( 'dsh-select2', plugins_url( 'assets/lib/select2/select2.min.js', DYNAMIC_SHORTCODES__FILE__ ), [ 'jquery' ], DYNAMIC_SHORTCODES_VERSION, true );
		wp_register_style(
			'dynamic-shortcodes-admin',
			DYNAMIC_SHORTCODES_URL . 'assets/css/admin.css',
			[],
			DYNAMIC_SHORTCODES_VERSION,
		);
		wp_enqueue_style( 'dynamic-shortcodes-admin' );

		wp_register_script(
			'dynamic-shortcodes-filters',
			DYNAMIC_SHORTCODES_URL . 'assets/js/filters.js',
			[],
			DYNAMIC_SHORTCODES_VERSION,
			true
		);

		wp_register_script(
			'dynamic-shortcodes-admin',
			DYNAMIC_SHORTCODES_URL . 'assets/js/admin.js',
			[ 'dsh-select2', 'dynamic-shortcodes-filters' ],
			DYNAMIC_SHORTCODES_VERSION,
			true
		);
		wp_enqueue_script( 'dynamic-shortcodes-admin' );
		wp_enqueue_style( 'dsh-select2' );

		wp_register_script(
			'dynamic-shortcodes-demo',
			DYNAMIC_SHORTCODES_URL . 'assets/js/demo.js',
			[ 'wp-util', 'jquery' ],
			DYNAMIC_SHORTCODES_VERSION,
			true
		);
		wp_register_style(
			'dynamic-shortcodes-demo',
			DYNAMIC_SHORTCODES_URL . 'assets/css/demo.css',
			[],
			DYNAMIC_SHORTCODES_VERSION,
			false
		);
		wp_enqueue_script( 'dynamic-shortcodes-demo' );
		wp_enqueue_style( 'dynamic-shortcodes-demo' );

		if ( self::MAIN_PAGE_ID . '_page_dynamic-shortcodes-collection' === $hook_suffix ) {
			wp_register_script(
				'dynamic-shortcodes-collection',
				DYNAMIC_SHORTCODES_URL . 'assets/js/collection.js',
				[ 'jquery', 'dynamic-shortcodes-filters' ],
				DYNAMIC_SHORTCODES_VERSION,
				true
			);
			wp_register_style(
				'dynamic-shortcodes-collection',
				DYNAMIC_SHORTCODES_URL . 'assets/css/collection.css',
				[],
				DYNAMIC_SHORTCODES_VERSION,
				'all'
			);

			wp_enqueue_script( 'dynamic-shortcodes-collection' );
			wp_enqueue_style( 'dynamic-shortcodes-collection' );
		}
	}

	public function add_menu_pages() {
		add_menu_page(
			DYNAMIC_SHORTCODES_PRODUCT_NAME,
			DYNAMIC_SHORTCODES_PRODUCT_NAME,
			'manage_options',
			self::MAIN_PAGE_ID,
			[ Plugin::instance()->settings_manager->admin_pages->settings, 'render_page' ],
			Plugin::instance()->plugin_utils->assets->get_logo_url_svg_onecolor_base64(),
			59
		);
		add_submenu_page(
			self::MAIN_PAGE_ID,
			DYNAMIC_SHORTCODES_PRODUCT_NAME . ' - ' . esc_html__( 'Settings', 'dynamic-shortcodes' ),
			esc_html__( 'Settings', 'dynamic-shortcodes' ),
			'manage_options',
			self::MAIN_PAGE_ID,
			[ Plugin::instance()->settings_manager->admin_pages->settings, 'render_page' ],
			1
		);
		add_submenu_page(
			self::MAIN_PAGE_ID,
			DYNAMIC_SHORTCODES_PRODUCT_NAME . ' - ' . esc_html__( 'Power Shortcodes', 'dynamic-shortcodes' ),
			esc_html__( 'Power Shortcodes', 'dynamic-shortcodes' ),
			'manage_options',
			'dynamic-shortcodes-power',
			[ Plugin::instance()->settings_manager->admin_pages->power_shortcodes, 'render_page' ],
			2
		);
		add_submenu_page(
			self::MAIN_PAGE_ID,
			DYNAMIC_SHORTCODES_PRODUCT_NAME . ' - ' . esc_html__( 'Getting Started', 'dynamic-shortcodes' ),
			esc_html__( 'Getting Started', 'dynamic-shortcodes' ),
			'manage_options',
			'dynamic-shortcodes-getting-started',
			[ Plugin::instance()->settings_manager->admin_pages->getting_started, 'render_page' ],
			3
		);
		add_submenu_page(
			self::MAIN_PAGE_ID,
			DYNAMIC_SHORTCODES_PRODUCT_NAME . ' - ' . esc_html__( 'Demo Shortcodes', 'dynamic-shortcodes' ),
			esc_html__( 'Demo Shortcodes', 'dynamic-shortcodes' ),
			'manage_options',
			'dynamic-shortcodes-demo',
			[ Plugin::instance()->settings_manager->admin_pages->demo, 'render_page' ],
			4
		);
		add_submenu_page(
			self::MAIN_PAGE_ID,
			DYNAMIC_SHORTCODES_PRODUCT_NAME . ' - ' . esc_html__( 'Collection', 'dynamic-shortcodes' ),
			esc_html__( 'Collection', 'dynamic-shortcodes' ),
			'manage_options',
			'dynamic-shortcodes-collection',
			[ Plugin::instance()->settings_manager->admin_pages->collection, 'render_page' ],
			5
		);
		add_submenu_page(
			self::MAIN_PAGE_ID,
			DYNAMIC_SHORTCODES_PRODUCT_NAME . ' - ' . esc_html__( 'License', 'dynamic-shortcodes' ),
			esc_html__( 'License', 'dynamic-shortcodes' ),
			'manage_options',
			Plugin::instance()->plugin_utils->get_license_admin_page(),
			[ Plugin::instance()->settings_manager->admin_pages->license, 'render_page' ],
			6
		);
	}
}
