<?php
/**
 * Woffice - Events Widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Events_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Events widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-events';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Events widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Events', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Events widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-fun-facts';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Events widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice Events content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

		$this->start_controls_section(
			'woffice_events_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_events_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_events_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Events', 'woffice' ),
				'placeholder' => esc_html__( 'Events', 'woffice' ),
			]
		);

        $visibility_option = $this->get_event_visiblity_list();

        $this->add_control(
			'woffice_events_visibility',
			[
				'label' => esc_html__( 'Event Visibility', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'personal',
				'options' => $visibility_option,
			]
		);

		$this->end_controls_section();

	}

    public function get_event_visiblity_list() {
        $group_options = [];
		if (woffice_bp_is_active('groups')) {
			$groups_query = BP_Groups_Group::get(array('show_hidden' => true,'update_meta_cache'  => false));
			foreach ($groups_query['groups'] as $group) {
				$group_options['group_' . $group->id] = $group->name;
			}
		}
		$args = array(
			'post_type'      => 'project',
			'posts_per_page' => '-1',
		);

		$user_posts      = get_posts($args);
		$project_options = array();
		foreach ($user_posts as $project) {
			$project_options['project_' . $project->ID] = $project->post_title;
		}

		$visibility = array(
			'personal' => __('Personal', 'woffice'),
			'general' => __('General', 'woffice'),
		);
        $visibility = array_merge($visibility,$group_options,$project_options);
        
        return $visibility;
    }

    /**
	 * Render [Job] output from WP Job Manager on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $title = $settings['woffice_events_title'];
        $event_visibility = $settings['woffice_events_visibility'];
        
		?>
		<div class="widget widget_woffice_event">
			<div class="box">
				<div class="intern-padding">
					<div class="intern-box box-title">
					<?php
						echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_events_title_tag'] ), $this->get_render_attribute_string( 'title' ), $title );
					?>
					</div>
                    <?php
                        $current_user = wp_get_current_user();

                        if ($event_visibility == 'personal' || $event_visibility == 'general') {
                            $visibility = $event_visibility;
                            $visibility_id = $current_user->ID;
                        } else {
                            $visibility_data = explode("_", $event_visibility);
                            $visibility = $visibility_data[0];
                            $visibility_id = $visibility_data[1];
                        }
                        
                        if ($current_user) {
                            echo do_shortcode("[woffice_calendar widget='true' visibility='$visibility' id='{$visibility_id}']");
                        }
                    ?>			
				</div>
			</div>
		</div>
		<?php
		$this->run_event_script();
	}

	public function run_event_script() {

		if ( \Elementor\Plugin::instance()->editor->is_edit_mode() === false ) {
			return;
		}

	?>
		<script type="text/javascript" src="<?php echo woffice_get_extension_uri('event', 'static/js/events.vue.js');?>"></script>
	<?php

	}
}