<?php
/**
 * Woffice - Fun Facts Widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Fun_Facts_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Fun Fact widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-fun-facts';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Fun Fact widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Fun Facts', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Fun Fact widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-fun-facts';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Fun Fact widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

		$this->start_controls_section(
			'woffice_funfacts_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);


		$this->add_control(
			'woffice_funfact_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Fun Facts', 'woffice' ),
				'placeholder' => esc_html__( 'Fun Facts', 'woffice' ),
			]
		);

		$this->add_control(
			'woffice_funfacts_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'fact_icon',
			[
				'label' => esc_html__( 'Icon', 'woffice' ),
				'type' => \Elementor\Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-heart',
					'library' => 'fa-solid',
				],
				'recommended' => [
					'fa-solid' => [
						'circle',
						'dot-circle',
						'square-full',
					],
					'fa-regular' => [
						'circle',
						'dot-circle',
						'square-full',
					],
				],
			]
		);

		$repeater->add_control(
			'fact_content',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'type' => \Elementor\Controls_Manager::WYSIWYG,
				'default' => esc_html__( 'List Content' , 'woffice' ),
				'show_label' => false,
			]
		);

		$this->add_control(
			'fun_facts',
			[
				'label' => esc_html__( 'Fun Facts Item', 'woffice' ),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'fact_icon' => 'fas fa-heart',
						'fact_content' => esc_html__( 'Anything you like here...Pick up an icon and set any content you wish. ', 'woffice' ),
					],
					[
						'fact_icon' => 'fas fa-bullhorn',
						'fact_content' => esc_html__( 'It can be announcements, alerts, jokes or anything else.', 'woffice' ),
					],
					[
						'fact_icon' => 'fas fa-hand-peace',
						'fact_content' => esc_html__( 'Choose in 650 icons to illustrate your idea, and use this widget in any sidebar. ', 'woffice' ),
					]
				],
				'title_field' => '{{{ fact_content }}}',
			]
		);

		$this->end_controls_section();

	}

    /**
	 * Render [Job] output from WP Job Manager on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$title = $settings['woffice_funfact_title'];
		?>
			<div class="widget_woffice_funfacts">
				<div class="box">
					<div class="widget_funfacts intern-box box-title">
						<?php echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_funfacts_title_tag'] ), $this->get_render_attribute_string( 'woffice_funfact_title' ), $title );?>
					</div>
					<div class="flexslider">
						<ul class="slides">
							<?php
								if($settings['fun_facts']) {
									foreach($settings['fun_facts'] as $value) {
							?>
								<li>
									<?php  \Elementor\Icons_Manager::render_icon( $value['fact_icon'], [ 'aria-hidden' => 'true' ] ); ?>
									<p> <?php echo sanitize_text_field($value['fact_content']);?> </p>
								</li>
							<?php
									}
								}
							?>
						</ul>
					</div>
				</div>
				<?php $this->render_flex_script();?>
			</div>
		<?php
	}

	protected function content_template() {
			
		?>
			<div class="widget_woffice_funfacts">
				<div class="box">
					<div class="widget_funfacts intern-box box-title">
						<{{{ settings.woffice_funfacts_title_tag }}}>{{{ settings.woffice_funfact_title }}}</{{{ settings.woffice_funfacts_title_tag }}}>
					</div>
					<div class="flexslider">
						<ul class="slides">
							<# _.each( settings.fun_facts, function( item ) { 
								var iconHTML = elementor.helpers.renderIcon( view, item.fact_icon, { 'aria-hidden': true }, 'i' , 'object' );	
							#>
								<li>
									{{{ iconHTML.value }}}
									<p>{{{item.fact_content}}}</p>
								</li>
							<# }); #>
						</ul>
					</div>
				</div>
			</div>
			<?php $this->render_flex_script();?>
		<?php
	}

	protected function render_flex_script() {
		if ( \Elementor\Plugin::instance()->editor->is_edit_mode() === false ) {
			return;
		}
		?>
			<script type="text/javascript">
				jQuery( document ).ready( function( $ ) {
					$('.widget_woffice_funfacts .flexslider').flexslider({
						animation: "slide",
						animationLoop: true,
						slideshow: true,
						directionNav: false,
						selector: ".slides > li",
						smoothHeight: false,
						start: function(){
							$('.widget_woffice_funfacts .flexslider').resize();
						}
					});
				});
			</script>
		<?php
	}
}