<?php
/**
 * Woffice - Project widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Projects_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Project widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-projects';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Project widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Projects', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Project widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-woffice-project';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Project widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    public function get_project_taxonomy(){

        $taxonomies = get_object_taxonomies( 'project', 'objects' );
		$data       = array();

		foreach ( $taxonomies as $tax_slug => $tax ) {
			if ( ! $tax->public || ! $tax->show_ui ) {
				continue;
			}

			$data[ $tax_slug ] = $tax;
		}

		return apply_filters( 'woffice_post_loop_taxonomies', $data, $taxonomies, 'project' );
    }

    /**
	 * Register Woffice Project content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

        $taxonomy = $this->get_project_taxonomy();

		$this->start_controls_section(
			'woffice_projects_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_project_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_projects_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Projects', 'woffice' ),
				'placeholder' => esc_html__( 'Projects', 'woffice' ),
			]
		);

        $this->add_control(
			'only_current_user',
			[
				'label' => esc_html__( 'Only current user', 'woffice' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'woffice' ),
				'label_off' => esc_html__( 'No', 'woffice' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

        $this->add_control(
			'project_status',
			[
				'label' => esc_html__( 'Status', 'woffice' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'all',
				'options' => [
					'all' => esc_html__( 'All', 'woffice' ),
                    'planned' => esc_html__( 'Planned', 'woffice' ),
                    'in_progress' => esc_html__( 'In progress', 'woffice' ),
					'in_review' => esc_html__( 'In review', 'woffice' ),
					'archived' => esc_html__( 'Archived', 'woffice' ),
					'done'  => esc_html__( 'Done', 'woffice' ),
					
				],
			]
		);

        if(!empty($taxonomy)){
            foreach ( $taxonomy as $index => $tax ) {
                $terms = get_terms( $index );

                $related_tax = array();

                if ( ! empty( $terms ) ) {
                    foreach ( $terms as $t_index => $t_obj ) {
                        $related_tax[ $t_obj->slug ] = $t_obj->name;
                    }
                    // Add control for all taxonomies.
                    $this->add_control(
                        'tax_' . $index . '_filter',
                        array(
                            /* translators: %s label */
                            'label'       => sprintf( __( '%s Filter', 'uael' ), $tax->label ),
                            'type'        => Controls_Manager::SELECT2,
                            'multiple'    => true,
                            'default'     => '',
                            'label_block' => true,
                            'options'     => $related_tax,
                        )
                    );
                }
            }
        }

		$this->end_controls_section();

	}

    /**
	 * Render [Job] output from WP Job Manager on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $widget_projects_query = $this->build_project_query($settings);
        $title = $settings['woffice_projects_title'];
        $user_id = false;
        if(isset($settings['only_current_user']) && $settings['only_current_user'] == 'yes') {
            $user_id = get_current_user_id();
        }
?>
    <!-- WIDGET -->
        <div class="widget_woffice_projects">
            <div class="box">
                <div class="widget_projects intern-box box-title">
                    <?php
						echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_project_title_tag'] ), $this->get_render_attribute_string( 'woffice_projects_title' ), $title );
					?>
                </div>
                <ul class="list-styled list-projects">
                    <?php

                            /**
                             * Filter the maximum number of projects to display in the widget "(Woffice) Recent Projects"
                             *
                             * @param int
                             */
                            $widget_projects_max = apply_filters('woffice_widget_recent_projects_max', 8);

                            $number_projects = 0;
                            
                            while ($widget_projects_query->have_posts()) : $widget_projects_query->the_post();
                        
                                if ($number_projects === $widget_projects_max)
                                    break;
                        
                                if ($user_id) {
                                    $project_members = ( function_exists( 'get_post_meta' ) ) ? get_post_meta(get_the_ID(), 'project_members',true) : '';
                        
                                    if (!empty($project_members) && !in_array($user_id, $project_members)) {
                                        continue;
                                    }
                                }
                        
                                echo'<li>';
                                    echo '<a href="'. get_the_permalink() .'" rel="bookmark">'. get_the_title() .'</a>
                                    <div class="progress-custom">
                                        <div class="progress">
                                            <div class="progress-bar" role="progressbar" aria-valuenow="' . woffice_projects_percentage() . '" aria-valuemin="0" aria-valuemax="100"  style="width: ' . woffice_projects_percentage() . '%"></div>
                                        </div>
                                        <div class="progress-value">
                                            ' . woffice_projects_percentage() . '%
                                        </div>
                                    </div>';
                                echo '</li>';
                        
                                $number_projects++;
                        
                            endwhile;
                        
                            if ($number_projects == 0)
                                esc_html_e("Sorry you don't have any project yet.","woffice");
                        
                            wp_reset_postdata();
                    ?>
                </ul>
            </div>
        </div>
<?php
	}

    public function build_project_query($settings) {
        $project_query_args = array(
            'post_type' => 'project',
            'posts_per_page' => '-1'
        );
    
        $projects = new WP_query($project_query_args);
        $excluded = array();
        $hide_projects_archived = woffice_get_theming_option( 'hide_projects_completed', true );
    
        while ( $projects->have_posts() ) : $projects->the_post();
    
            $hide = ( $hide_projects_archived ) ? (bool)get_post_meta( get_the_ID(), 'project_completed', true) : false;
    
            if(!woffice_is_user_allowed_projects() || $hide) {
                array_push($excluded, get_the_ID());
            }
        endwhile;
    
        // QUERY $tax
        $query_args = array(
            'post_type' => 'project',
            'post__not_in' => $excluded,
            'posts_per_page' => -1,
        );
        $category = isset($settings['tax_project-category_filter']) && !empty($settings['tax_project-category_filter']) ? $settings['tax_project-category_filter'] : false;
        if (!empty($category) && $category !== "all") {
            $the_tax = array(array(
                'taxonomy' => 'project-category',
                'terms' => $category,
                'field' => 'slug',
            ));
            $query_args['tax_query'] = $the_tax;
        }

        $status = isset( $settings['project_status']) ? $settings['project_status'] : 'all';
    
        if (!empty($status) && $status !== 'all') {
            $query_args['meta_key']     = 'project_current_status';
            $query_args['meta_compare'] = 'LIKE';
    
            if ($status == 'archived') {
                $query_args['meta_value'] = 'archived';
            } else if ($status === 'done') {
                $query_args['meta_value'] = 'done';
            } else if ($status === 'in_progress') {
                $query_args['meta_value'] = 'in_progress';
            } else if ($status === 'in_review') {
                $query_args['meta_value'] = 'in_review';
            } else if ($status === 'planned') {
                $query_args['meta_value'] = 'planned';
            }
        }

        /**
         * Filter the query args for the filter "(Woffice) Recent Projects"
         *
         * @param array $query_args
         */
        $query_args = apply_filters('woffice_widget_recent_projects_query_args', $query_args);
        $widget_projects_query = new WP_Query( $query_args );

        return $widget_projects_query;
    }

}