<?php
/**
 * Woffice - Wiki Widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Wikis_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Wiki widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-wiki';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Wiki widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Wiki', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Wiki widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-wiki';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Wiki widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice Wiki content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

        $taxonomy = $this->get_wiki_taxonomy();

		$this->start_controls_section(
			'woffice_wiki_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_wiki_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_wiki_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Wiki', 'woffice' ),
				'placeholder' => esc_html__( 'Wiki', 'woffice' ),
			]
		);

        $this->add_control(
			'number_of_wiki',
			[
				'label' => esc_html__( 'Number of Wiki', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 6,
			]
		);

        if(!empty($taxonomy)){
            foreach ( $taxonomy as $index => $tax ) {
                $terms = get_terms( $index );

                $related_tax = array();

                if ( ! empty( $terms ) ) {
                    foreach ( $terms as $t_index => $t_obj ) {
                        $related_tax[ $t_obj->slug ] = $t_obj->name;
                    }
                    // Add control for all taxonomies.
                    $this->add_control(
                        'tax_' . $index . '_filter',
                        array(
                            /* translators: %s label */
                            'label'       => sprintf( __( '%s Filter', 'uael' ), $tax->label ),
                            'type'        => Controls_Manager::SELECT2,
                            'multiple'    => true,
                            'default'     => '',
                            'label_block' => true,
                            'options'     => $related_tax,
                        )
                    );
                }
            }
        }


		$this->end_controls_section();

	}

    public function get_wiki_taxonomy(){

        $taxonomies = get_object_taxonomies( 'wiki', 'objects' );
		$data       = array();

		foreach ( $taxonomies as $tax_slug => $tax ) {
			if ( ! $tax->public || ! $tax->show_ui ) {
				continue;
			}

			$data[ $tax_slug ] = $tax;
		}

		return apply_filters( 'woffice_post_loop_taxonomies', $data, $taxonomies, 'wiki' );
    }

    public function build_woffice_wiki_query($settings) {

        $posts_per_page = $settings['number_of_wiki'];

        $query_args = array(
            'post_type' => 'wiki',
            'posts_per_page' => $posts_per_page,
            'post_status' => array( 'publish'),
        );

        $category = $settings['tax_wiki-category_filter'];

        if (!empty($category) && $category != "all") {

            $the_tax = array(array(
                'taxonomy' => 'wiki-category',
                'terms' => $category,
                'field' => 'slug',
            ));

            $query_args['tax_query'] = $the_tax;
        }

        // GET WIKI POSTS
        $widget_wiki_query = new WP_Query($query_args);

        return $widget_wiki_query;
    }

    /**
	 * Render output from wiki on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $title = $settings['woffice_wiki_title'];
        $widget_wiki_query = $this->build_woffice_wiki_query($settings);
        ?>
            <div class="widget widget_woffice_wiki">
                <div class="box">
                    <div class="widget_wiki intern-box box-title">
                        <?php
                            echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_wiki_title_tag'] ), $this->get_render_attribute_string( 'woffice_wiki_title' ), $title );
                        ?>
                    </div>
                    <!-- WIDGET -->
	                <ul class="list-styled list-wiki pl-0">
                        <?php 
                            // GET WIKI POSTS
                            while( $widget_wiki_query->have_posts()) : $widget_wiki_query->the_post();

                                if (woffice_is_user_allowed_wiki()){
                                    $likes = woffice_get_wiki_likes(get_the_ID());
                                    $likes_display = (!empty($likes)) ? $likes : '';
                                    echo'<li class="is-'.get_post_status().' pt-2"><a href="'. get_the_permalink() .'" rel="bookmark" data-post-id="'.get_the_ID().'">'. get_the_title() . '</a>'. $likes_display.'</li>';
                                }

                            endwhile;
                            wp_reset_postdata();
                        ?>
                    </ul>
                </div>    
            </div>
		<?php

	}
}