<?php

/**
 * The file that defines the core plugin actions
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://woffice.io
 * @since      1.0.0
 *
 * @package    Woffice_Credit_System
 * @subpackage Woffice_Credit_System/includes
 */

class Woffice_Credit_System_Actions {

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */

    public function __construct() {
        add_action('wp_ajax_nopriv_woffice_ai_request', array($this,'content_ai_request'));
        add_action('wp_ajax_woffice_ai_request', array($this,'content_ai_request'));
        add_action('wp_ajax_nopriv_woffice_ai_add_todos', array($this,'add_ai_todos'));
        add_action('wp_ajax_woffice_ai_add_todos', array($this,'add_ai_todos'));
    }

    /**
	 * Request the AI action and recive the response from AI
	 *
	 * @since    1.0.0
     * @return   HTML OR Error message
	 */
    public function content_ai_request() {
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : false;
        $rq_type = isset($_POST['rq_type']) ? sanitize_text_field($_POST['rq_type']) : false;
        $api_url = WOCS_CSAI_URL;
        $api_key = get_option('wocs_csai');
        $api_status = get_option('wocs_csai_status');
        $result = array();

        if ( !wp_verify_nonce( $_POST['nonce'], 'ajax-nonce' ) ) {
            wp_die( __('Sorry! Direct Access is not allowed.', 'woffice-credits'));
        }

        if($api_status == 'deactivated') {

            $result = array(
                'success' => false,
                'output' => __('Please activate your API key.','woffice-credits'),
                'credit' => 0,
            );

            echo wp_json_encode($result);            
            exit();
        }
   
        if(! $this->wocs_check_ai_auth()) {
            $result = array(
                'success' => false,
                'output' => __("Sorry! You don't have valid permission to perform this action.",'woffice-credits'),
                'credit' => 0,
            );

            echo wp_json_encode($result);            
            exit;
        }
        
        $args = array(
            'method'  => 'POST',
            'timeout' => 60,
            'body'    => array(
                'api_key' => $api_key,
                'title' => $title,
                'rq_type' => $rq_type,
            ),
        );

        $ai_request = wp_remote_post($api_url . 'content_ai',$args);
        $response = json_decode(wp_remote_retrieve_body($ai_request));

        if(!empty($response) && is_object($response) && $response->success) {

            $result = array(
                'success' => true,
                'output' => $this->wocs_generate_ai_output($response->output,$rq_type),
                'credit' => $response->available_credit,
            );

            update_option('wocs_csai_avail_credit',$response->available_credit);

        } else {

            $result = array(
                'success' => false,
                'output' => $response->message,
            );
        }
        echo wp_json_encode($result);
        exit();
    }

    /**
	 * Check if user has valid permission or not
	 *
	 * @since    1.0.0
     * @return   bool
	 */

    public function wocs_check_ai_auth() {
        $output = true;
    
        if ( ! current_user_can( 'edit_posts') ) {
            $output = false;
         }

         return $output;
    }

    /**
	 * Generate html for AI response
	 *
	 * @since    1.0.0
     * @return   HTML 
	 */
    public function wocs_generate_ai_output($data,$rq_type){
        $html = '';
        
        if($rq_type == 'title') {
            if(!empty($data) && is_array($data)) {
                $html .= "<ul class='wocs_ai_list'>";
                    foreach($data as $key => $reuslt){
                        $html .= sprintf('<li class="wpcs_ai_item"><span class="wocs_ai_title">%1$s</span><span class="pl-3"><a class="small wpcs_copier" href="javascript:void(0)">%2$s</a></span></li>',$reuslt,__('Use as title','woffice-credits'));
                    }
                $html .= "</ul>";
            }
        } if($rq_type == 'todo') {
            if(!empty($data) && is_array($data)) {
                $html .= '<form method="post" name="ai_todos_form" class="ai_todos_form">';
                    $html .= "<ul class='wocs_ai_list'>";
                        $html .= '<div class="form-check p-2"><label><input class="mr-3" type="checkbox" id="selectll_ai_todos"/>'.__('Select all','woffice').'</label></div>';
                        foreach($data as $key => $reuslt){
                            $result = preg_replace('/\d+/u', '', $reuslt);
                            $result = str_replace('.','',$result);
                            $html .= sprintf('<li class="wpcs_ai_item"><div class="form-check"><input class="form-check-input" type="checkbox" name="ai_todos[]" value="%1$s"><label class="form-check-label pl-2" for="flexCheckDefault">%1$s</label></div></li>',$result);
                        }
                    $html .= "</ul>";
                    $html .= '<button type="button" class="btn btn-success ai_add_todos">'.__('Add todos','woffice-credit').'</button>';
                $html .= "</form>";
            }
        } else if($rq_type == 'project_content' || $rq_type == 'blog_content') {
            $html .= sprintf('<span class="wocs_ai_content">');

            foreach($data as $key => $reuslt){
                $html .= sprintf(wpautop($reuslt));
            }

            $html .= '</span>  <a href="javascript:void(0)" class="d-block btn-more">'.__('Read more...','woffice-credits').'</a><a class="wpcs_content_copier" href="javascript:void(0)">'.__('Use This','woffice-credits').'</a>';
        }else if($rq_type == 'blog_image') {
            $html .= "<ul class='wocs_ai_list' id='thumbs'>";
            foreach($data as $key => $reuslt){
                $html .= sprintf('<li class="wpcs_ai_item thumb"><img class="wocs_ai_image" width="140" height="140" src="%1$s"><span class="pl-3"><a class="small wpcs_image_copier" href="javascript:void(0)">%2$s</a></span></li>',$reuslt,__('Use this image','woffice-credits'));
            }
        }
        return $html;
    }

    /**
	 * Check available credit for user.
	 *
	 * @since    1.0.0
     * @return   Update option
	 */
    public function content_ai_check_credit() {

        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : false;
        $rq_type = isset($_POST['rq_type']) ? sanitize_text_field($_POST['rq_type']) : false;
        $api_url = WOCS_CSAI_URL;
        $api_key = get_option('wocs_csai');
        $api_status = get_option('wocs_csai_status');
        
        $args = array(
            'method'  => 'POST',
            'timeout' => 60,
            'body'    => array(
                'api_key' => $api_key,
            ),
        );

        $ai_request = wp_remote_post($api_url . 'content_ai_getcredit', $args);
        $response = json_decode(wp_remote_retrieve_body($ai_request));

        if(!empty($response) && is_object($response) && $response->success) {

            $credit = $response->output;
            update_option('wocs_csai_avail_credit',$credit);
        } 
    }

    public function add_ai_todos(){

        $todos = isset($_POST['todos']) ? $_POST['todos'] : false;
        $args = [];
        $parse_todos = wp_parse_str($todos,$args);

        if(!check_ajax_referer('woffice_todos') || !isset($_POST['id']) || !isset($_POST['type'])) {
            echo json_encode(array('status' => 'fail'));
            die();
        }

        // We get the ID from the current Project post
        $id = intval($_POST['id']);

        $project_todo = function_exists('woffice_get_project_todos_lists') ? woffice_get_project_todos_lists($id) : '';

        $ai_todos = isset($args['ai_todos']) && !empty($args['ai_todos']) ? $args['ai_todos'] : false;
        $current_user = get_current_user_id();
        $now = new DateTime();
        $time_zone  = date_default_timezone_get();
        $now->setTimeZone(new DateTimezone($time_zone));
        $new_todos = array();

        if($ai_todos) {
            foreach ($ai_todos as $key => $todo) {
                $new_task = array(
                    'title' => $todo,
                    'note' => '',
                    'start_date' => date("Y-m-d H:i:s"),
                    'date' => date("Y-m-d H:i:s", strtotime('+24 hours')),
                    'urgent' => '',
                    'assigned' => '',
                    'has_kanban' => '',
                    'review_request' => '',
                    'todo_reviewer' => '',
                    'done' => '',
                    'todo_comments' => array(),
                );
                array_push($project_todo['project_todo_lists'],$new_task);
            }   
            
            $updated_list = $project_todo;

        }

        $update_post_option = function_exists('woffice_update_project_todos_lists') ? woffice_update_project_todos_lists($id,$updated_list) : false;
        
        $response = array();
        
        if($update_post_option) {
            $response = array(
                'status' => true,
                'message' => __('Todo added successfully.','woffice')
            );
        } else {
            $response = array(
                'status' => false,
                'message' => __('Todo not added.','woffice')
            );
        }

        // We return a success to let our user know
        echo json_encode($response);

        die();
    }
}