<?php
/**
 * Woffice - Birthday Widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Birthday_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Birthday widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-birthday';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Birthday widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Birthday', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Birthday widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-birthday';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Birthday widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice Birthday content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

		$this->start_controls_section(
			'woffice_birthday_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_birthday_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_birthday_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Birthday', 'woffice' ),
				'placeholder' => esc_html__( 'Birthday', 'woffice' ),
			]
		);

		$this->end_controls_section();

	}

    /**
	 * Render [Job] output from WP Job Manager on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $title = $settings['woffice_birthday_title'];
        $birthdays    = $this->woffice_birthdays_get_array();
    
		?>
		<div class="widget widget_woffice_birthday">
			<div class="box">
				<div class="intern-padding">
					<div class="intern-box box-title">
                        <?php
                            echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_birthday_title_tag'] ), $this->get_render_attribute_string( 'title' ), $title );
                        ?>
                    </div>
                    <ul class="birthdays-list">
                        <?php
                            /**
                             * Before the list birthdays, in the birthdays widget
                             */
                            do_action('woffice_birthdays_widget_before_birthdays_list');

                            $this->woffice_birthdays_content($birthdays);

                            /**
                             * After the list birthdays, in the birthdays widget
                             */
                            do_action('woffice_birthdays_widget_after_birthdays_list');
                        ?>
                    </ul>
				</div>
			</div>
		</div>
		<?php
	}

    	/**
	 * This function returns an array of the members birthdays :
     * Only today and upcoming birthdays sorted in ascending order
     *
	 * array(
	 * 	  id_member => array('datetime' => DateTime Object);
	 * )
	 */
	public function woffice_birthdays_get_array() {

		$woffice_wp_users = get_users(array('fields' => array('ID'),'cache_results' => false));
		$members_birthdays = array();

		// Get the Birthday field name
		$field_option = get_option('fw_ext_settings_options:woffice-birthdays');
        $field_name = isset($field_option['birthday_field_name']) ? $field_option['birthday_field_name'] : false;
		$field_name = str_replace( "'", "\'", $field_name);

		// Get the Birthday field ID
		$field_id = xprofile_get_field_id_from_name( $field_name );

		// Set all data for the date limit check
        $birthdays_limit = isset($field_option['birthdays_range_limit']) ? $field_option['birthdays_range_limit'] : false;
        $max_date = $this->woffice_birthday_range_limit_date($birthdays_limit);

		// We check if the member has a birthday set
		foreach ($woffice_wp_users as $woffice_wp_user) {

            $birthday_string = maybe_unserialize(BP_XProfile_ProfileData::get_value_byid( $field_id, $woffice_wp_user->ID));

			if (empty($birthday_string)) {
				continue;
			}

			// We transform the string in a date
			$birthday = DateTime::createFromFormat('Y-m-d H:i:s', $birthday_string);

			/**
			 * Filter if the current birthday (in the birthdays widget) can be displayed
			 *
			 * @param bool $is_displayed
			 * @param int $user_id
			 * @param DateTime $birthday
			 */
			$display_this_birthday = apply_filters('woffice_display_this_birthday', true, $woffice_wp_user->ID, $birthday);

			if ($birthday !== false && $display_this_birthday) {

                // Skip if birth date is not in the selected limit range
                if (!$this->woffice_birthday_is_in_range_limit($birthday, $max_date)) {
                    continue;
                }

				$celebration_year = (date('md', $birthday->getTimestamp()) >= date('md') ) ? date('Y') : date('Y', strtotime('+1 years') );

			    $years_old = (int) $celebration_year - (int)date("Y", $birthday->getTimestamp());

			    // If gone for this year already, we remove one year
			    if (date('md', $birthday->getTimestamp()) > date('md')) {
					$years_old = $years_old;
				} else if (date('md', $birthday->getTimestamp()) == date('md')) {
				    $years_old = $years_old;
			    }

				/**
				 * Filter `woffice_birthdays_date_format`
				 *
				 * Let you change the date format in which the birthday is displayed
				 * See: http://php.net/manual/en/function.date.php
				 *
				 * @param string - the date format PHP value
				 *
				 * @return string
				 */
				$format = apply_filters('woffice_birthdays_date_format', 'md');

			    $celebration_string = $celebration_year . date($format, $birthday->getTimestamp());

				$members_birthdays[$woffice_wp_user->ID] = array(
                    'datetime' => $birthday,
					'next_celebration_comparable_string' => $celebration_string,
					'years_old' => $years_old
				);
			}

		}

        uasort($members_birthdays, array($this, "date_comparison"));

		return $members_birthdays;
	}

    	/**
	 * It will generate the content for the front viw
	 *
	 * @param array $all_bithdays
	 */
	public function woffice_birthdays_content($all_bithdays) {

		if (empty($all_bithdays)) {
			return;
		}
        $max_items = woffice_get_theming_option('birthdays_to_display');
        $birthday_age = woffice_get_theming_option('display_age');
        $birthday_date_format = woffice_get_theming_option('birthday_date_format');
        $c = 0;

		$date_ymd = date('Ymd');

        foreach($all_bithdays as $user_id => $birthday) {
            if ($c === $max_items)
                break;

            $activation_key = get_user_meta($user_id, 'activation_key');
            if (empty($activation_key)) {
                $name_to_display = woffice_get_name_to_display($user_id);

				$age = $birthday["years_old"];

				// We don't display negative ages
				$mem_domain = function_exists('bp_members_get_user_url') ? bp_members_get_user_url($user_id) : bp_core_get_user_domain($user_id);
				if($age > 0) {
					echo '<li class="clearfix">';
					if (function_exists('bp_is_active')):
						echo '<a href="' . $mem_domain . '">';
						echo get_avatar($user_id);
						echo '</a>';
					else :
						echo get_avatar($user_id);
					endif;
					echo '<span class="birthday-item-content">';
					echo '<strong>' . $name_to_display . '</strong>';
					if ($birthday_age != 'nope') {
						echo ' <i>(' . $age . ')</i>';
					}
					echo ' ', _x('on', 'happy birthday ON 25-06', 'woffice');
					$date_format = $birthday_date_format;
					$date_format = (!empty($date_format)) ? $date_format : 'F d';

					if ( 'F d' === $date_format ) {
						//To make months translatable
						echo ' <span class="badge badge-primary badge-pill">';
						_e(date('F', $birthday["datetime"]->getTimestamp()), 'woffice');
						echo ' ';
						echo date('d', $birthday["datetime"]->getTimestamp());
						echo '</span>';
					} else {
						echo ' <span class="badge badge-primary badge-pill">' . date_i18n( $date_format, $birthday["datetime"]->getTimestamp() ) . '</span>';
					}

					$happy_birthday_label = '';
					if($birthday["next_celebration_comparable_string"] == $date_ymd)
						$happy_birthday_label = '<span class="badge badge-primary badge-pill">' . __( 'Happy Birthday!', 'woffice') . '</span>';

					/**
					 * The label "Happy birthday", if today is the birthday of an user
					 *
					 * @param string $happy_birthday_label The text of the label (contains some HTML)
					 * @param int $user_id
					 */
					echo apply_filters( 'woffice_today_happy_birthday_label', $happy_birthday_label, $user_id );

					echo '</span>';
					echo '</li>';

                    $c++;
				}
            }

        }
	}

    /**
     * Check if given birth day is within the range
     *
     * @param string $birth_date
     * @param string $max_date
     *
     * @return boolean
     */
    public function woffice_birthday_is_in_range_limit($birth_date, $max_date) {
        if ($max_date == 'all') {
            return true;
        }

        $target_date = date('md', $birth_date->getTimestamp());
        $now_date = date('md');

        return $max_date >= $target_date && $target_date >= $now_date;
    }

    /**
     * Get the max date(dayMonth) we will show for the birthday
     *
     * @param string $birthdays_limit
     *
     * @return string
     */
    public function woffice_birthday_range_limit_date($birthdays_limit) {

        if ($birthdays_limit == 'monthly') {
            $int_date_time = strtotime('+30 day', time());
            return date('md', $int_date_time);
        }

        if ($birthdays_limit == 'weekly') {
            $int_date_time = strtotime('+7 day', time());
            return date('md', $int_date_time);
        }
        return 'all';
    }

    /**
     * Used for order array of object, containing dates
     *
     * @param array $a
     * @param array $b
     *
     * @return boolean
     */
    private function date_comparison($a, $b) {
        return strcmp($a['next_celebration_comparable_string'] , $b['next_celebration_comparable_string']);
    }
}