<?php
/**
 * Woffice - Poll Widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Poll_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Poll widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-poll';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Poll widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Poll', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Poll widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-poll';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Welcome widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice Poll content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

		$this->start_controls_section(
			'woffice_poll_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_poll_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_poll_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Poll', 'woffice' ),
				'placeholder' => esc_html__( 'Poll', 'woffice' ),
			]
		);

		$this->add_control(
			'woffice_poll_type',
			[
				'label' => esc_html__( 'Type', 'woffice' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'answers',
				'options' => [
					'answers'  => esc_html__( 'Answers', 'textdomain' ),
					'question' => esc_html__( 'Question', 'textdomain' ),
				],
				
			]
		);

		$this->end_controls_section();

	}

	/**
	 * GENERATE RESULTS FRONTEND
	 */
	public function woffice_poll_get_results() {
	
		global $wpdb;
		$poll_option = woffice_get_theming_option('woffice_poll_answers');
		echo '<div id="woffice-poll-result">';
			$answers = $poll_option;
			if(!empty($answers)){
				foreach ($answers as $answer) {
					echo '<div class="woffice-poll-result-answer">';
						// We check first that the table exists
						$check = $wpdb->get_results("SELECT 1 FROM ".WOFFICE_POLL_TABLE." LIMIT 1");
						if ($check != FALSE) {
							$reps = $wpdb->get_var( $wpdb->prepare( "SELECT reps FROM ".WOFFICE_POLL_TABLE." WHERE label = %s",$answer) );
							$all_reps = $wpdb->get_var( "SELECT SUM(reps) FROM ".WOFFICE_POLL_TABLE );
							$rep_percentage = ($all_reps!=0) ? round(($reps/$all_reps)*100,1) : 0;
							echo '<p>'.esc_html($answer).'</p>';
								echo '<div class="progress-custom">
								<div class="progress">
									<div class="progress-bar" role="progressbar" aria-valuenow="' . esc_attr($rep_percentage) . '" aria-valuemin="0" aria-valuemax="100"  style="width: ' .esc_attr($rep_percentage) . '%"></div>
								</div>
								<div class="progress-value">
									' . esc_attr($rep_percentage) . '%
								</div>
							</div>';
						}
					echo'</div>';
				}
			}
		echo'</div>';
		
	}

    /**
	 * Render [Job] output from WP Job Manager on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $title = $settings['woffice_poll_title'];
		$poll_option = woffice_get_theming_option('woffice_poll_answers');
		$answers = $poll_option;
		$name = woffice_get_theming_option('woffice_poll_question');
		$type = isset($settings['woffice_poll_type']) ? $settings['woffice_poll_type'] : false;
		?>
		<div class="widget woffice_widget_poll">
			<div class="box">
				<div class="intern-padding">
					<div class="intern-box box-title">
					<?php
						echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_poll_title_tag'] ), $this->get_render_attribute_string( 'title' ), $title );
					?>
					</div>			
					<div class="textwidget">
						<?php 
							if(!empty($answers)){
								/**
								 * LOAD THE FORM
								 */
								echo '<form id="woffice_poll" action="'.$_SERVER['REQUEST_URI'].'" method="POST" class="wpcf7">';
									
									// Only for logged users ? 
									$logged_only = woffice_get_theming_option( 'woffice_poll_logged_only' );
									if ($logged_only == true && !is_user_logged_in()) {
										
										echo '<div class="woffice-poll-ajax-reply">';
											echo'<i class="fa fa-lock"></i>
											<p>'.__("Sorry ! It is only for logged users.","woffice").'</p>';
										echo '</div>';
										
									}
									else {
										
										/* CHECK IF USER ALREADY VOTED */
										$user_ID = get_current_user_id();
										$the_saved_users = get_option('woffice_poll_users');
										$the_saved_users_array = (is_array($the_saved_users)) ? $the_saved_users : array();
										if( !in_array($user_ID,$the_saved_users_array) && $type == "question"){
											/* QUESTION NAME */
											echo '<p>'.esc_html($name).'</p>';
											/*NOT DONE YET 0.9 */
											echo '<span class="wpcf7-checkbox">';
												/* LOOP ANSWER */
												foreach ($answers as $answer) {
													echo '<span class="wpcf7-list-item">';
														echo '<label class="poll-answer">';
															/*CHECK DATA KIND*/
																echo '<input type="checkbox" name="poll[]" id="'.esc_attr($answer).'" value="'.esc_attr($answer).'">';
															/*ANSWER*/
															echo '<span class="wpcf7-list-item-label">'. esc_html($answer) .'</span>';
														echo '</label>';
													echo '</span>';
												}
											echo'</span>';
											
											/*SUBMIT BUTTON*/
											echo'<p class="text-right">';        
												echo wp_nonce_field( 'woffice_poll_nonce','woffice_poll_nonce_field');
												echo'<input type="hidden" name="action" value="woffice_poll_add_answer"/>';
												echo'<button type="submit" class="btn btn-default">';
													echo'<i class="fa fa-arrow-right"></i>';
													 _e( 'Send', 'woffice');
												 echo'</button>';
											echo'</p>';
											
										}
										else {
											/*DISPLAY RESULTS*/
											echo'<div class="woffice-poll-ajax-reply sent">
												<i class="fa fa-check"></i>
												<p>'.__("Thanks ! Here are the results for","woffice").'</p>
												<span class="poll-question-back">'.$name.'</span>
											</div>'; 
											$this->woffice_poll_get_results();
										}
									
									}
									
								echo '</form>';
								
								/**
								 * MESSAGE FROM AJAX 
								 */
								echo'<div id="woffice_ajax_validation"></div>' ;
								
							}
							else {
								echo '<p>'. __('Sorry, there is no answer set yet in the Extension options.','woffice') .'</p>';
							}
						?>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}