<?php
/**
 * Woffice - Tasks Widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Tasks_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Tasks widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-tasks';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Tasks widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Tasks', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Tasks widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-user-tasks';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Tasks widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice Tasks content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

		$this->start_controls_section(
			'woffice_tasks_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_tasks_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_tasks_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'You have {no_of_tasks} tasks', 'woffice' ),
				'placeholder' => esc_html__( 'You have {no_of_tasks} tasks', 'woffice' ),
			]
		);


		$this->end_controls_section();

	}

    /**
	 * Return array list of the assigned tasks for a certain user
     *
     * @param $user_ID
     * @return array
	 */
	public function woffice_projects_assigned_tasks($user_ID) {

		/*Array of assigned tasks*/
		$the_assigned_tasks = array();
		/*Counter*/
		$count = 0;
		
		if ($user_ID != 0) {
		
			/*We loop all the projects to fetch tasks*/
			$projects_query = new WP_Query('post_type=project&showposts=-1');
			while($projects_query->have_posts()) : $projects_query->the_post(); 
					
				/*We get the tasks*/
				$project_tasks = woffice_get_project_todos_lists(get_the_ID());
				if (!empty($project_tasks)) {
					
					/*We loop the task*/
					foreach ($project_tasks as $task){

						if(!isset($task['assigned']))
							continue;
						
						/* We check if it's not done AND it's assigned to the user */
						$task['assigned'] = (is_array($task['assigned'])) ? $task['assigned'] : explode(',',$task['assigned']);
						if (isset($task['done']) && $task['done'] == false && in_array($user_ID, $task['assigned'])){
							
							$title_task = (!empty($task['title'])) ? $task['title'] : "";
							$title_date = (!empty($task['date'])) ? $task['date'] : "";
							
							$the_assigned_tasks[] = array(
								'task_name' => $title_task,
								'task_date' => $title_date,	
								'task_project' => get_permalink(),
							);
							$count++;
							
						}
					
					}
				}
				
			endwhile;
			wp_reset_postdata();
			
		}

		return array( 'number' => $count , 'tasks' => $the_assigned_tasks); 
		
	}

    /**
	 * Render [Job] output from WP Job Manager on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $title = $settings['woffice_tasks_title'];
        $taks_list = $this->woffice_projects_assigned_tasks(get_current_user_id());
        $title = str_replace('{no_of_tasks}',$taks_list['number'],$title);
		?>
            <div class="widget widget_woffice_tasks_assigned">
                <div class="box">
                    <div class="project-assigned-container">
                        <div class="project-assigned-head">
                            <div class="box-title">
                                <?php
                                    echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_tasks_title_tag'] ), $this->get_render_attribute_string( 'woffice_tasks_title' ), $title );
                                ?>
                            </div>
                        </div>
                        <?php /* We get the tasks */
                            if (isset($taks_list['tasks']) && !empty($taks_list['tasks'])) { ?>
                                <ul class="assigned-tasks-list pl-0">
                                    <?php 
                                    foreach ($taks_list['tasks'] as $task){
                                        echo '<li class="assigned-task">';
                                            echo '<a href="'.$task['task_project'].'?#project-content-todo">';
                                                if (!empty($task['task_date'])) {
                                                    echo '<span class="badge badge-primary badge-pill mr-2">'. date(get_option('date_format'),strtotime(esc_html($task['task_date']))) .'</span>';
                                                }
                                                echo $task['task_name'];
                                            echo'</a>';
                                        echo '</li>';
                                    }
                                    ?>
                                </ul>
                            <?php } else { ?>
                                <div class="assigned-tasks-empty">
                                    <p><?php _e("Well done! You don't have any task from your projects.","woffice"); ?></p>
                                </div>
                        <?php } ?>
                    </div> 
                </div>    
            </div>
		<?php

	}
}