<?php 

defined( 'ABSPATH' ) || exit;

// Include the WP_List_Table if it doesn't already exist
if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class Woffice_User_Registration_Actions_Table extends WP_List_Table {
    public function __construct() {
        parent::__construct([
            'singular' => __('User', 'woffice'),
            'plural'   => __('Users', 'woffice'),
            'ajax'     => false,
        ]);
    }

    public static function get_users($status_filter = '', $per_page = 5, $page_number = 1) {

        $page_number = max(1, intval($page_number));
        $per_page = max(1, intval($per_page));

        $meta_query = [];
        if ($status_filter) {
            $meta_query[] = [
                'key'     => 'woffice_user_role_status',
                'value'   => $status_filter,
                'compare' => '='
            ];
        } else {
            $meta_query[] = [
                'key'     => 'woffice_user_role_status',
                'value'   => '',
                'compare' => '!='
            ];
        }

        $args = [
            'role'       => 'subscriber',
            'orderby'    => 'user_login',
            'order'      => 'ASC',
            'number'     => $per_page,
            'offset'     => ($page_number - 1) * $per_page,
            'meta_query' => $meta_query
        ];

        return get_users($args);
    }

    public static function record_count($status_filter = '') {
        $meta_query = [];
        if ($status_filter) {
            $meta_query[] = [
                'key'     => 'woffice_user_role_status',
                'value'   => $status_filter,
                'compare' => '='
            ];
        } else {
            $meta_query[] = [
                'key'     => 'woffice_user_role_status',
                'value'   => '',
                'compare' => '!='
            ];
        }

        $args = [
            'role'       => 'subscriber',
            'meta_query' => $meta_query
        ];

        $users = get_users($args);
        return count($users);
    }

    public function no_items() {
        _e('No users available.', 'woffice');
    }

    public function column_default($item, $column_name) {
        $user_status = get_user_meta($item->ID,'woffice_user_role_status',true);
        
        switch ($column_name) {
            case 'username':
                $avatar = get_avatar($item->ID, 32);
                $edit_link = get_edit_user_link($item->ID);
                return sprintf(
                    '%s <strong>%s</strong><br><a href="%s" class="edit-link">Edit</a>',
                    $avatar,
                    $item->user_login,
                    esc_url($edit_link)
                );
			case 'useremail':
				return sprintf('<a href="mailto:%s">%s</a>',sanitize_email($item->user_email),sanitize_email($item->user_email));
            case 'role_request':
                $user_role_request = get_user_meta($item->ID,'woffice_user_role',true);
                if(!in_array($user_role_request,array('administrator','super_admin','super'))){
                    return sprintf('%s',str_replace('_',' ',$user_role_request));
                }
            case 'status':
                return sprintf('%s',$user_status);
            case 'actions':
                $approve_url = add_query_arg(['action' => 'approve', 'user_id' => $item->ID]);
                $reject_url = add_query_arg(['action' => 'reject', 'user_id' => $item->ID]);
                $delete_url = add_query_arg(['action' => 'delete', 'user_id' => $item->ID]);
                
                switch ($user_status) {
                    case 'approved':
                        return sprintf(
                            '<a href="%s">'.__('Reject','woffice').'</a> | <a href="%s">Delete</a>',
                            esc_url($reject_url),
                            esc_url($delete_url)
                        );
                    case 'rejected':
                        return sprintf(
                            '<a href="%s">'.__('Approve','woffice').'</a> | <a href="%s">Delete</a>',
                            esc_url($approve_url),
                            esc_url($delete_url)
                        );
                    case 'deleted':
                        return sprintf(
                            '<a href="%s">'.__('Approve','woffice').'</a> | <a href="%s">Reject</a>',
                            esc_url($approve_url),
                            esc_url($reject_url)
                        );
                    default:
                        return 'Unknown status';
                }
            default:
                return print_r($item, true); // Show the whole array for troubleshooting purposes
        }
    }

    public function get_bulk_actions() {
        $actions = [
            'bulk-approve' => 'Approve',
            'bulk-reject'  => 'Reject',
            'bulk-delete'  => 'Delete',
        ];
        return $actions;
    }

    public function process_bulk_action() {
        if (isset($_POST['user_ids']) && is_array($_POST['user_ids'])) {
            $action = $this->current_action();
            foreach ($_POST['user_ids'] as $user_id) {
                if (!current_user_can('edit_user', $user_id)) {
                    continue;
                }

                switch ($action) {
                    case 'bulk-approve':
                        update_user_meta($user_id, 'woffice_user_role_status', 'approved');
                        break;
                    case 'bulk-reject':
                        update_user_meta($user_id, 'woffice_user_role_status', 'rejected');
                        break;
                    case 'bulk-delete':
                        delete_user_meta($user_id, 'woffice_user_role_status', 'deleted');
                        break;
                }
            }
        }
    }

    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="user_ids[]" value="%s" />',
            $item->ID
        );
    }

    public function get_columns() {
        $columns = [
            'cb'       => '<input type="checkbox" />',
            'username' => __('Username', 'woffice'),
			'useremail' => __('Email', 'woffice'),
            'role_request' => __('Role Request', 'woffice'),
            'status' => __('Status', 'woffice'),
            'actions'  => __('Actions', 'woffice'),
        ];
        return $columns;
    }

    public function prepare_items() {
        $this->process_bulk_action();

        $status_filter = isset($_POST['status_filter']) ? sanitize_text_field($_POST['status_filter']) : '';

        $per_page     = $this->get_items_per_page('users_per_page', 5);
        $current_page = $this->get_pagenum();
        $total_items  = self::record_count($status_filter);

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
        ]);

        $this->items = self::get_users($status_filter, $per_page, $current_page);

        $columns  = $this->get_columns();
        $hidden   = [];
        $sortable = [];

        $this->_column_headers = [$columns, $hidden, $sortable];
    }

    public function extra_tablenav($which) {
        if ($which == "top") {
            $status_filter = isset($_POST['status_filter']) ? sanitize_text_field($_POST['status_filter'] ) : '';
            ?>
            <div class="alignleft actions">
                <label for="filter-by-status" class="screen-reader-text"><?php _e('Filter by status', 'woffice'); ?></label>
                <select name="status_filter" id="filter-by-status">
                    <option value=""><?php _e('All statuses', 'woffice'); ?></option>
                    <option value="approved" <?php selected($status_filter, 'approved'); ?>><?php _e('Approved', 'woffice'); ?></option>
                    <option value="rejected" <?php selected($status_filter, 'rejected'); ?>><?php _e('Rejected', 'woffice'); ?></option>
                    <option value="deleted" <?php selected($status_filter, 'deleted'); ?>><?php _e('Deleted', 'woffice'); ?></option>
                </select>
                <?php
                submit_button(__('Filter'), '', 'filter_action', false);
                ?>
            </div>
            <?php
        }
    }
}
